"""
Единая для внешних и внутренних вики обёртка для работы с группами

@author: chapson
"""

import logging
from typing import List

from django.conf import settings

from wiki.org import get_org, get_org_dir_id
from wiki.users.models import Group
from wiki.intranet.models import Staff
from wiki.utils.per_request_memoizer import memoize_for_this_request

IS_INTRANET = getattr(settings, 'IS_INTRANET', False)

logger = logging.getLogger(__name__)

if IS_INTRANET:
    from wiki.intranet.models import GroupMembership
    from wiki.pages.access.cache import all_group_ids_cached

    user_group_ids = all_group_ids_cached

    class WikiServiceHasGone(Exception):
        pass

    def get_wiki_service_group():
        return Group.objects.get(service_id=287)

    ADMIN_GROUP_MEMBERSHIP_STAFF_IDS = set([])

    def admin_group_staff_ids():
        """
        Вернуть set из staff_id из админской группы.

        @rtype: set
        """
        if not ADMIN_GROUP_MEMBERSHIP_STAFF_IDS:
            ADMIN_GROUP_MEMBERSHIP_STAFF_IDS.update(
                {
                    group_membership.staff_id
                    for group_membership in GroupMembership.objects.filter(group_id=settings.WIKI_ADMIN_GROUP_ID)
                }
            )
        return ADMIN_GROUP_MEMBERSHIP_STAFF_IDS

    def drop_admin_membership_cache():
        """
        Нужно исключительно для тестов.
        """
        ADMIN_GROUP_MEMBERSHIP_STAFF_IDS.clear()

    @memoize_for_this_request
    def is_admin(user):
        if not user.is_authenticated:
            return False

        # кешируется
        return user.staff.id in admin_group_staff_ids()

    _yandex_group = None

    @memoize_for_this_request
    def is_tech_admin(user):
        return is_admin(user)

    def yandex_group():
        """Вернуть группу "Яндекс", которая всегда существует"""
        global _yandex_group
        if _yandex_group is None:
            _yandex_group = Group.objects.get(id=settings.YANDEX_GROUP_ID)
            if _yandex_group.url != 'yandex':
                raise ValueError('Yandex group changed ID, something is fundamentally wrong')

        return _yandex_group

    _yamoney_group = None

    def yamoney_group():
        """Вернуть группу "Яндекс-Деньги" или None"""
        global _yamoney_group
        if _yamoney_group is None:
            try:
                _yamoney_group = Group.objects.filter(id=51)[0]
            except IndexError:
                _yamoney_group = False

        if _yamoney_group is False:
            # работать все еще можно, но это очень плохая ситуация.
            logger.error('You don\'t have Yandex.Money group! Something is wrong!')
            return None
        else:
            logger.debug('Yandex.Money group is "%s"', _yamoney_group)

    def in_wiki_external(staff):
        """
        Вернуть True, если сотрудник входит в вики-группу wiki_external.
        """
        return any(group.id == 48408 for group in staff.all_groups)

    def true_if_wiki_external_is_common(staff):
        """
        Вернуть True если члены вики-группы wiki_external имеют доступ до Вики как сотрудники Яндекса.
        """
        if settings.WIKI_CODE != 'wiki':
            return False
        return in_wiki_external(staff)

    def common_access_is_ok(staff):
        """
        Применим ли доступ "COMMON".

        @type staff: Staff
        """
        return any(group.name == settings.IDM_ROLE_EMPLOYEE_GROUP_NAME for group in user_django_groups(staff.user))

    @memoize_for_this_request
    def user_django_groups(user):
        return user.groups.all()


else:

    @memoize_for_this_request
    def user_group_ids(staff: Staff) -> List[int]:
        return [group.id for group in staff.user.get_all_groups()]

    def yandex_group():
        """Вернуть фальшивку, в которой внешний пользователь никогда состоять не может"""
        return None

    def yamoney_group():
        """Вернуть фальшивку, в которой внешний пользователь никогда состоять не может"""
        return None

    def get_wiki_service_group():
        """
        Not really wiki service for extranet.
        Rename must be done.
        """
        if settings.IS_BUSINESS:
            return Group.objects.get(group_dir_type='organization_admin', org=get_org())
        else:
            return Group.objects.get(name__in=['Admins', 'admins'])

    @memoize_for_this_request
    def is_admin(user):
        """Возвращает, состоит ли пользователь в группе администраторов"""
        if user.is_authenticated:
            if settings.IS_BUSINESS:
                return Group.objects.filter(
                    user__in=[user], group_dir_type='organization_admin', org=get_org()
                ).exists()
            else:
                return Group.objects.filter(user__in=[user], name__in=['Admins', 'admins']).exists()
        return False

    @memoize_for_this_request
    def is_tech_admin(user):
        if settings.IS_BUSINESS:
            return get_org_dir_id() == settings.ETALON_ORG_DIR_ID
        return is_admin(user)

    def drop_admin_membership_cache():
        return

    def user_django_groups(user):
        return []
