"""
Модуль для получения raw_access

В вики введено условное понятие "raw_access". Это словарь соответствующий странице, которая необязательно существует:
r = {
        #Супертег этой страницы
        'supertag': string,

        #Список супертегов этой страницы, порядок неважен
        'supertags': list,

        #Полный список прав (модель Access), отсортированный по убыванию.
        #От самой страницы к корню.
        'full_list':  list,

        #Права доступа, которые относятся к одной странице: к данной или ближайшему существующему родителю.
        #Такая логика вызвана наследованием прав доступа.
        'list': list,

        #Длина списка list
        'list_len': 0,

        #Длина списка full_list_len
        'full_list_len': 0,

        #супертег ближайшей существующей страницы или данной, если она существует.
        'latest_supertag': None,
        }
raw_access интерпретируется функцией interpret_raw_access, которая выдает информацию о правах доступа к этой странице.

@author: chapson
"""

from collections import defaultdict

from django.conf import settings
from django.db.models.query import QuerySet  # noqa
from wiki.legacy.url import Url

from wiki.org import get_org
from wiki.pages.models import Access, Page
from wiki.utils.supertag import tag_to_supertag

IS_INTRANET = getattr(settings, 'IS_INTRANET', False)


def get_raw_access(tag_or_page):
    """
    Получить словарь с сырыми правами доступа к странице по ее тэгу
    @type tag_or_page: Page|basestring

    @rtype: dict
    @return: Структура с сырыми данными по правам доступа к странице
    """
    if isinstance(tag_or_page, str):
        supertag = tag_to_supertag(tag_or_page)
        try:
            page = Page.active.get(supertag=supertag, org=get_org())
        except Page.DoesNotExist:
            # FIXME: иногда в тестах сюда передается несуществующий supertag,
            # при этом ожидается валидная, хоть и пустая, структура с правами.
            # get_bulk_raw_access в такой ситуации вернет пустой dict, потому подсовываем stub
            supertags = Url.get_ancestors(supertag)
            supertags.append(supertag)
            return {
                'supertag': supertag,
                'supertags': supertags,
                'full_list': [],
                'full_list_len': 0,
                'list': [],
                'list_len': 0,
                'latest_supertag': None,
            }
    else:
        page = tag_or_page

    return get_bulk_raw_access([page])[page]


def filter_rules(supertags, supertags_to_access_rules):
    """
    Вернуть только те правила доступа, которые относятся к заданным супертегам.

    @type supertags: list
    @type supertags_to_access_rules: dict
    """
    result = []
    for supertag in supertags:
        result += supertags_to_access_rules.get(supertag, [])
    return result


def get_access_by_supertags(supertags):
    """Выбрать из базы список всех прав доступа"""
    # список прав доступа сразу ко всем страницам и их предкам одним запросом,
    # сортирован по -supertag страницы
    related_fields = ['page', 'staff', 'group']
    if settings.IS_INTRANET:
        related_fields.append('staff__department')
    return (
        Access.objects.select_related(*related_fields)
        .filter(page__supertag__in=supertags, page__org=get_org())
        .order_by('-page__supertag')
    )


def populate_access_data(page, supertags_to_access_rules, force_inheritance: bool = False):
    """
    Заполняет структуру с данными по доступу для конткретной страницы

    @type page: Page
    @param page: объект страницы
    @type supertags_to_access_rules: dict
    @param supertags_to_access_rules: Словарь {supertag -> [access_rules]} - супертэга в список его правил.

    @rtype: dict
    @return: словарь с правами доступа страницы
    """
    # Супертэги в этом списке должны быть упорядочены от потомков к предкам.
    # Это важно для корректного извлечения правил доступа, относящихся к
    # указанной странице, а не к ее предку.

    supertags = [page.supertag] + Url.get_ancestors(page.supertag)

    if force_inheritance:
        # как если бы у самой страницы не было ACL вообще
        # -- обычно нужно чтобы прикинуть от кого будем наследоваться
        # и показать это в интерфейсе
        supertags = supertags[1:]

    full_list = filter_rules(supertags, supertags_to_access_rules)

    # Супертэг первого предка, для которого определены правила доступа
    # (или супертэг самой страницы, если для нее определен доступ).
    first_existing_supertag = full_list and full_list[0].page.supertag or None

    list_ = []
    # Collect all Access objects for the first existing page (since they are
    # sorted, we can break immediately when we see another page).
    for access in full_list:
        if access.page.supertag == first_existing_supertag:
            list_.append(access)
        else:
            break
    return {
        'supertag': page.supertag,
        'supertags': supertags,
        'full_list': full_list,
        'full_list_len': len(full_list),
        'list': list_,
        'list_len': len(list_),
        'latest_supertag': first_existing_supertag,
    }


def get_bulk_raw_access(pages, force_inheritance: bool = False):
    """
    Получить словарь с сырыми правами доступа к страницам

    !!! ВНИМАНИЕ !!!
    В эту функцию нельзя передавать сразу слишком много страниц, т.к. она строит
    словарь с их правами доступа, который может легко забить всю память.

    @type pages: list | QuerySet
    @param pages: список или QuerySet страниц

    @rtype: dict
    @return: словарь с правами доступа для каждой переданной страницы.

    [Page, Page, ...] --> dict(supertag=raw_access, ...)
    """
    result = {}  # dict под структуры прав доступа для каждой страницы

    # супертеги страниц и их всех возможных родителей
    supertags = set(p.supertag for p in pages)
    add_all_ancestors(supertags)

    # права доступа по супертегам
    full_access_list = list(get_access_by_supertags(supertags))

    # для wiki-biz расширяем модель Group данными об иерархии групп
    if settings.IS_BUSINESS:
        for access in full_access_list:
            if access.group:
                access.group = access.group.group

    supertags_to_access_rules = defaultdict(list)
    for rule in full_access_list:
        supertags_to_access_rules[rule.page.supertag].append(rule)

    # заполняем структуру с данными по доступу для каждой страницы
    for page in pages:
        result[page] = populate_access_data(page, supertags_to_access_rules, force_inheritance)

    return result


def add_all_ancestors(supertags_set):
    """
    Добавить в список супертэгов всех родителей всех объектов.

    @type supertags_set: set
    """
    processed = set()
    missing = supertags_set - processed
    while missing:
        page = missing.pop()
        ancestors = [page] + Url.get_ancestors(page)
        processed.update(ancestors)
        supertags_set.update(ancestors)
        missing = missing - processed
