from wiki.api_frontend.utils.accessors import department_groups_by_ids
from wiki.pages.models.consts import AclType
from wiki.api_v2.public.pages.page.schemas import AclUpdateSchema, NOT_SET
from wiki.pages.access import get_bulk_raw_access, interpret_raw_access
from wiki.pages.access.consts import LegacyAcl
from wiki.pages.models import Page
from wiki.users.dao import get_users_by_identity
from wiki.users.logic import groups_by_ids
from django.conf import settings


def convert_acl_request_to_legacy_acl(a: AclUpdateSchema) -> LegacyAcl:
    acl = LegacyAcl(acl_type=a.acl_type, is_readonly=a.is_readonly, break_inheritance=a.break_inheritance)

    if a.acl_type != AclType.CUSTOM:
        return acl

    if a.users is not NOT_SET:
        acl.users = get_users_by_identity(a.users, prefetch_staff=True, panic_if_missing=True, apiv1_exceptions=False)
    else:
        acl.users = []

    if a.groups is not NOT_SET:
        group_models, _ = groups_by_ids(a.groups)
        acl.groups = list(group_models)
    else:
        acl.groups = []

    if not settings.IS_INTRANET and a.departments is not NOT_SET:
        acl.groups += list(department_groups_by_ids(a.departments))

    return acl


def get_legacy_acl(page: Page, force_inheritance: bool = False) -> LegacyAcl:
    """
    :param page:
    :param force_inheritance: Принудительно включить наследование (игнорируем ACL у текущего, если был)
    :return:
    """
    raw_access = get_bulk_raw_access([page], force_inheritance)[page]
    access = interpret_raw_access(raw_access)
    acl_type = AclType.DEFAULT

    if access['is_restricted']:
        acl_type = AclType.CUSTOM

    if access['is_owner']:
        acl_type = AclType.ONLY_AUTHORS

    inherited_from_root = len(raw_access['list']) == 0 and raw_access['latest_supertag'] is None

    inherited = bool(raw_access['list']) and raw_access['latest_supertag'] != raw_access['supertag']

    return LegacyAcl(
        acl_type=acl_type,
        break_inheritance=not inherited and not inherited_from_root,
        is_readonly=page.is_readonly,
        inherits_from=raw_access['latest_supertag'] if inherited else None,
        users=[u.user for u in access['users']],
        groups=access['groups'],
    )
