
from django import forms
from django.contrib import admin
from django.contrib.auth import get_user_model

from wiki.intranet.models.intranet_extensions import Group
from wiki.pages.models import Page
from wiki.pages.models.intranet_extensions import Officiality


class OfficialityEditForm(forms.ModelForm):

    page = forms.ModelChoiceField(
        queryset=Page.active.all(),
        to_field_name='supertag',
        widget=forms.TextInput,
    )

    responsible_persons = forms.CharField(
        help_text='Логины, разделенные пробелами или переносами строк',
        widget=forms.Textarea,
        required=False,
    )
    responsible_groups = forms.CharField(
        help_text='Коды групп, разделенные пробелами или переносами строк',
        widget=forms.Textarea,
        required=False,
    )

    def __init__(self, *args, **kwargs):
        instance = kwargs.get('instance')
        if instance:
            # сохраним и используем в OfficialityAdmin.save_model
            self.page_old = instance.page

            initial = {
                'page': instance.page.supertag,
                'responsible_persons': '\n'.join([person.username for person in instance.responsible_persons.all()]),
                'responsible_groups': '\n'.join([person.url for person in instance.responsible_groups.all()]),
            }
            kwargs['initial'] = initial
        super(OfficialityEditForm, self).__init__(*args, **kwargs)

    class Meta:
        model = Officiality
        fields = (
            'page',
            'responsible_persons',
            'responsible_groups',
        )

    def clean(self):
        cleaned_data = super(OfficialityEditForm, self).clean()

        responsible_persons = cleaned_data.get('responsible_persons')
        responsible_groups = cleaned_data.get('responsible_groups')

        if not responsible_persons and not responsible_groups:
            msg = 'Воу, воу, воу. Кто-то должен быть ответственным'
            raise forms.ValidationError(msg)

        return cleaned_data

    def _clean_space_delimited_field(self, form_field, model, db_field):
        data = self.cleaned_data[form_field]
        ids = data.split()
        objects = model.objects.filter(**{db_field + '__in': ids})

        fetched_ids = [getattr(obj, db_field) for obj in objects]
        for id in ids:
            if id not in fetched_ids:
                msg = 'Bad {field} {value}'.format(
                    field=db_field,
                    value=id,
                )
                raise forms.ValidationError(msg)

        return objects

    def clean_responsible_persons(self):
        return self._clean_space_delimited_field(
            form_field='responsible_persons',
            model=get_user_model(),
            db_field='username',
        )

    def clean_responsible_groups(self):
        return self._clean_space_delimited_field(
            form_field='responsible_groups',
            model=Group,
            db_field='url',
        )


class OfficialityAdmin(admin.ModelAdmin):
    form = OfficialityEditForm

    readonly_fields = (
        'created',
        'modified',
    )

    # поддержка денормализации в Page.is_official
    # пока официальность редактируется только в админке этот код может
    # оставаться тут, однако если появятся другие места изменения модели
    # Officiality, то код нужно будет перенести куда-то на сигналы модели.
    def save_model(self, request, obj, form, change):
        page_new = obj.page

        if change:
            page_old = form.page_old

            if page_new.pk != page_old.pk:
                self.mark_page_not_official(page_old)
                self.mark_page_official(page_new)
        else:
            self.mark_page_official(page_new)
        super(OfficialityAdmin, self).save_model(request, obj, form, change)

    def delete_model(self, request, obj):
        self.mark_page_not_official(obj.page)
        super(OfficialityAdmin, self).delete_model(request, obj)

    @staticmethod
    def mark_page_official(page):
        page.is_official = True
        page.save()

    @staticmethod
    def mark_page_not_official(page):
        page.is_official = False
        page.save()
