from django.conf import settings
from django.contrib import admin, messages

from wiki.pages.utils.resurrect import resurrect_page, RestorePageError


def restore_pages_action(modeladmin, request, queryset):
    """
    Восстановить все выбранные страницы
    """
    for page in queryset:
        try:
            resurrect_page(page)
        except RestorePageError as exc:
            messages.error(request, str(exc))


restore_pages_action.short_description = 'Восстановить выбранные страницы'


class PageAdmin(admin.ModelAdmin):
    """
    Настройки показа страниц в админке
    """

    #
    # Список объектов
    #

    # поля, показываемые в списке страниц
    list_display = (
        'id',
        'supertag',
        'page_type',
        'status',
        'formatter_version',
        'created_at',
        'title',
    )

    # по кому искать из поисковой строки
    search_fields = (
        '=id',
        'supertag',  # по супертэгу ищем как ILIKE
        '=tag',
    )

    list_per_page = 40

    # фильтры сбоку
    list_filter = (
        'page_type',
        'formatter_version',
        'lang',
        'is_documentation',
    )
    # иерархический фильтр по датам в шапке
    date_hierarchy = 'created_at'

    # по клику на каких полях переходить на редактирование страницы
    list_display_links = (
        'id',
        'supertag',
    )

    #
    # Форма редактирования объекта
    #

    # исключить из формы редактирования страницы
    if not settings.IS_INTRANET:
        exclude = ('is_official',)
    else:
        exclude = ()

    # readonly поля на форме редактирования
    readonly_fields = (
        'tag',
        'mds_storage_id',
        'supertag',
        'last_author',
        'page_type',
        'comments',
        'files',
        'opened_to_external_flag',
        'formatter_version',
        'lang',
        'redirects_to',
    )

    # добавить в действия "восстановить выбранные страницы"
    actions = (restore_pages_action,)

    def get_actions(self, request):
        """
        Удалить из списка экшнов пункт "удалить выбранные"
        """
        actions = super(PageAdmin, self).get_actions(request)
        if 'delete_selected' in actions:
            del actions['delete_selected']
        return actions
