from datetime import timedelta

from celery.utils.log import get_task_logger
from pymongo.errors import PyMongoError

# Default value for notification timeout in minutes
from wiki.api_v2.exceptions import AlreadyExists
from wiki.favorites_v2.logic import page_changed
from wiki.notifications.models import PageEvent
from wiki.org import get_org
from wiki.pages.logic import subscription as legacy_subscription
from wiki.pages.logic import tags as tags_logic
from wiki.pages.models import Page, PageWatch, Revision
from wiki.pages.utils.request import prepare_request
from wiki.subscriptions import logic as new_subscriptions
from wiki.users.logic import uses_new_subscriptions
from wiki.utils import timezone
from wiki.utils.diff import simple_diff
from wiki.utils.supertag import tag_to_supertag
from wiki.utils.wfaas.client import get_wfaas_client

DEFAULT_EVENT_TIMEOUT = 20

logger = get_task_logger(__name__)


def save_page(
    page_or_url,
    unicode_text,
    title=None,
    section_id=None,
    authors=None,
    request=None,
    user=None,
    update_modified=True,
    create_revision=True,
    page_type=None,
):
    """
    Сохранить какую-то вики страницу.

    Использовать для изменения вики-разметки страницы.

    Если при создании страницы передан невалидный тег, функция попытается его
    починить, в теории это должно всегда успешно получиться. В непредвиденных
    случаях райзится wiki.utils.errors.ValidationError.
    """
    if not request:
        request = prepare_request(page_or_url, user)

    is_new_page = False

    if isinstance(page_or_url, str):
        tag = page_or_url.strip('/')
        supertag = tag_to_supertag(tag)

        try:
            # Check if page already exists
            page = Page.active.get(supertag=supertag, org=get_org())
            # Не пишем поверх гридов
            assert page.page_type in (Page.TYPES.PAGE, Page.TYPES.WYSIWYG)
        except Page.DoesNotExist:
            tag = tags_logic.clean_tag(tag)
            supertag = tag_to_supertag(tag)
            page = Page(
                tag=tag,
                supertag=supertag,
                owner=request.user,
                org=get_org(),
                with_new_wf=True,
                wf_switched_at=None,
                wf_switched_author=None,
            )
            is_new_page = True
            if title is None:
                page.title = page.tag
            if page_type is not None:
                page.page_type = page_type
    else:
        page = page_or_url

    current_body = page.body
    current_title = page.title

    if section_id:
        unicode_text = replace_section(section_id, page.body, unicode_text)

    page.body = unicode_text

    new_body = page.body
    new_title = (title or current_title) or ''

    if not create_revision and not _was_page_changed(current_body, current_title, new_body, new_title):
        return page, None, False

    if update_modified:
        page.last_author = request.user
        page.modified_at = timezone.now()

    if title:
        page.title = title

    page.save()

    if is_new_page:
        if authors:
            page.authors.add(*authors)
        else:
            page.authors.add(request.user)

    revision = None
    if create_revision:
        revision = Revision.objects.create_from_page(page)

    try:
        page_changed(page, was_created=is_new_page)

    except PyMongoError:  # При недоступной монге не падать, а сохранить страницу без метаданных
        logger.exception('Mongo error')

    page.is_new_page = is_new_page
    return page, revision, True


def replace_section(section_number, page_body, section_body):
    ast = get_wfaas_client().raw_to_ast(page_body)
    section = ast.get_section(section_number)
    if section.pos_start is None:
        # Секция не найдена
        return page_body

    # Оставляем только один перевод строки в конце
    section_body = section_body.strip() + '\n'

    page_body = page_body[: section.pos_start] + section_body + page_body[section.pos_end :]

    return page_body


def get_page_section_data(page_body, section_number):
    ast = get_wfaas_client().raw_to_ast(page_body)
    section = ast.get_section(section_number)
    if section.pos_start is None:
        # Секция не найдена
        return

    return page_body[section.pos_start : section.pos_end]


def page_toc(page, user_ticket):
    toc = {}
    if page.page_type == Page.TYPES.PAGE:  # формируем toc только для обычных страниц

        wfaas_settings = {
            'page_path': page.supertag,
            'remark': {
                # Инклюды нужны, потому что toc может строиться из вложенных инклюдов
                'woofmd': {'actions': [{'name': 'include'}]},
            },
        }
        bemjson = get_wfaas_client().raw_to_bemjson(page.body, wfaas_settings, user_ticket)
        toc_data = bemjson.get('toc', {})
        toc_data = [toc_item for toc_item in toc_data if toc_item['type'] == 'heading']

        # Адаптируем toc под формат старого форматера,
        # чтобы не поломать логику текущих потребителей
        toc = {
            'block': 'wiki-doc',
            'content': [
                {
                    'block': 'wiki-toc',
                    'wiki-attrs': {'page': page.url},
                }
            ],
        }

        if not toc_data:
            return toc

        toc_items = []
        # В старом форматере level всегда начинался с единицы
        # в новом depth показывает реальный уровень заголовка.
        # Найдем смещение, которое нужно добавить, чтобы было
        # как в старом форматере
        min_level = min([toc_item.get('depth', 6) for toc_item in toc_data])
        level_offset = min_level - 1

        for toc_item in toc_data:
            toc_items.append(
                {
                    'block': 'wiki-tocitem',
                    'wiki-attrs': {
                        'txt': toc_item.get('value', ''),
                        'level': toc_item.get('depth', 6) - level_offset,
                        'anchor': toc_item.get('anchor', ''),
                    },
                }
            )

        toc['content'][0]['content'] = toc_items

    return toc


def create_event(is_new, page, revision=None, timeout=DEFAULT_EVENT_TIMEOUT, notify=True):
    """
    Create log's event after page has been updated.
    """
    event = PageEvent(
        page=page,
        author=page.last_author or page.get_authors().first(),
        timeout=page.modified_at + timedelta(minutes=timeout),
        event_type=PageEvent.EVENT_TYPES.create if is_new else PageEvent.EVENT_TYPES.edit,
        notify=True if notify and revision else False,
        created_at=page.modified_at,
    )

    event.meta = {'revision_id': revision.id if revision else None}

    if not is_new:
        diff = simple_diff(page._old_body, page.body)
        event.meta.update(
            {
                'added': '\n'.join(diff[0]),
                'deleted': '\n'.join(diff[1]),
            }
        )

    event.save()


def subscribe_to_new_page(page: Page, subscribe_author: bool):
    """
    Subscribe to new page all users who are watching for the nearest
    ancestor page as a cluster.
    """

    legacy_subscription.inherit_watches(page=page, exclude_users=list(page.get_authors()))

    if subscribe_author:
        author = page.last_author

        if author.is_robot():
            return  # -- robots dont have subscriptions --

        if uses_new_subscriptions(author):
            try:
                new_subscriptions.create_subscription(page=page, user=author, is_cluster=False)
            except AlreadyExists:
                pass
        else:
            legacy_subscription.create_watch(page=page, user=author, is_cluster=False)


def toggle_author_subscription(page, subscription):
    """
    Subscribe "page.last_author" to page's changes if "subscription" is True,
    and unsubscribe if it is False.
    """
    if subscription:
        PageWatch.objects.get_or_create(page=page, user=page.last_author.username)
    else:
        PageWatch.objects.filter(page=page, user=page.last_author.username).delete()


def _was_page_changed(current_body, current_title, new_body, new_title):
    return current_body != new_body or current_title != new_title
