"""
Перенос кластеров Вики.

Основную работу выполняет класс Cluster.

@author: chapson
"""

import logging
import re

from django.conf import settings
from django.contrib.auth import get_user_model
from django.db import IntegrityError, transaction
from django.db.models import Count, Q
from django.utils.translation import ugettext_lazy
from lxml import etree

from wiki.org import get_org, org_user
from wiki.pages import signals as page_signals
from wiki.pages.access import (
    ACCESS_COMMON,
    ACCESS_DENIED,
    ACCESS_RESTRICTED,
    get_access_status,
    get_raw_access,
    interpret_raw_access,
)
from wiki.pages.constants import ReservedSupertagAction as Action
from wiki.pages.dao import page as page_dao
from wiki.pages.dao.subscription import get_users_subscribed_to_page
from wiki.pages.logic import hierarchy, subscription
from wiki.pages.models import AbsentPage, Access, Page, PageLink, Revision
from wiki.pages.models.consts import ACTUALITY_STATUS
from wiki.pages.reserved_supertags import is_reserved_supertag
from wiki.users.core import get_support_contact
from wiki.users.logic import uses_new_subscriptions
from wiki.utils import timezone
from wiki.utils.backports.mds_compat import APIError
from wiki.utils.dict_keypath import value_by_path
from wiki.utils.models import queryset_iterator
from wiki.utils.path import canonize, resolve
from wiki.utils.supertag import translit
from wiki.utils.url import is_valid_url, is_wiki_url
from wiki.utils.wfaas.client import get_wfaas_client

User = get_user_model()

logger = logging.getLogger(__name__)
debug_logger = logging.getLogger('django.db')
notpage_link_template = re.compile('^(file|mailto):')


def make_link_markup(node):
    link_text = '(('
    link_text += node.get('url', '')

    children = node.get('children', [])
    for child_node in children:
        node_type = child_node.get('type', '')
        node_value = child_node.get('value', None)
        if node_type == 'text' and node_value is not None:
            link_text += f' {node_value}'
            break

    link_text += '))'
    return link_text


def get_straightened_link(link_string, _from, to, top_cluster, top_cluster_to, with_children, url=None):
    """
    Если линк ссылается на одну из переносимых страниц
    адаптируем его под новый адрес
    """
    if url is None:
        url = canonize(resolve(_from, link_string))

    # ссылка относительная
    if any(link_string.startswith(mark) for mark in ('.', '!')) or not link_string.startswith('/'):
        if not (url.startswith(top_cluster + '/') or url == top_cluster):  # ведет вовне кластера
            return url
        elif not with_children:  # ведет внутрь кластера и переносим без детей
            return url
        elif with_children:  # внутрь кластера и с детьми
            return canonize(resolve('/' + to, link_string))
        return None
    elif url.startswith(top_cluster + '/') and with_children:  # абсолютная ссылка ведет вовнутрь кластера
        return '/' + top_cluster_to + url[len(top_cluster) :]
    elif url == top_cluster:
        return '/' + to
    return None


def straighten_relative_links(text, user, _from, to, top_cluster, top_cluster_to, with_children):
    _from = '/' + _from
    top_cluster = '/' + top_cluster

    changed = False

    ast = get_wfaas_client().raw_to_ast(text)
    for node in reversed(list(ast.filter(['womLink', 'link']))):
        node_url = node.get('url', '')
        is_external_link = is_valid_url(node_url) and not is_wiki_url(node_url)

        if is_external_link or notpage_link_template.match(node_url):
            # Внешняя ссылка или ссылка не на страницу (email, файл)
            continue

        url_changed = get_straightened_link(node_url, _from, to, top_cluster, top_cluster_to, with_children)
        if url_changed:
            pos_start = value_by_path('position.start.offset', node)
            pos_end = value_by_path('position.end.offset', node)
            node['url'] = url_changed
            text = text[:pos_start] + make_link_markup(node) + text[pos_end:]
            changed = True

    return text, changed


def process_links(page_to_move, user, dest_supertag, top_cluster, top_cluster_to, with_children):
    """
    @type page_to_move: Page
    """
    try:
        page_to_move.body
    except APIError as exc:
        logger.exception('Cannot read from storage "%s"', exc)
        return

    body, has_changes = straighten_relative_links(
        text=page_to_move.body,
        user=user,
        _from=page_to_move.supertag,
        to=dest_supertag,
        top_cluster=top_cluster,
        top_cluster_to=top_cluster_to,
        with_children=with_children,
    )

    if has_changes:
        page_to_move.body = body

    return has_changes and body


def make_destination_tag(source, destination, with_children):
    """
    Для определения списка тегов страниц,
    которые должны появиться на месте dest из подстраниц дерева source
    """
    full_source = [source]
    if with_children:
        full_source += list(
            Page.objects.filter(
                supertag__startswith=source + '/',
                redirects_to__isnull=True,
                org=get_org(),
            ).values_list('supertag', flat=True)
        )
    destination_tags = []
    for source_tag in full_source:
        part = source_tag[len(source) :].strip('/')
        if part:
            part = '/' + part
        destination_tags.append(destination + part)
    return destination_tags


def any_page_reserved(clusters, with_children):
    """Все страницы списка clusters и подстраницы-нередиректы должны пройти проверку"""
    clusters_to_check = list(clusters)

    if with_children:
        condition = None
        for source in (translit(c) for c in clusters):
            c = Q(supertag__startswith=translit(source) + '/')
            if condition is None:
                condition = c
                continue
            condition = condition | c
        for page in Page.objects.filter(
            condition,
            redirects_to__isnull=True,
            org=get_org(),
        ).values_list('supertag', flat=True):
            clusters_to_check.append(page)

    return [page for page in clusters_to_check if is_reserved_supertag(page, Action.DELETE)]


def destination_reserved(clusters, with_children):
    clusters_to_check = []
    for source, destination in clusters.items():
        clusters_to_check += make_destination_tag(source, destination, with_children)
    return [supertag for supertag in clusters_to_check if is_reserved_supertag(supertag, Action.CREATE)]


def check_same_parent(supertags):
    """
    Check all the supertags have the same parent
    """
    parent = '/'.join(supertags[0].split('/')[:-1]) + '/'
    if parent == '/':
        parent = ''  # for second level nodes
    cnt = supertags[0].count('/')
    for supertag in supertags:
        if supertag.count('/') != cnt or not supertag.startswith(parent):
            return supertag, parent
    return None


def all_clusters_exist(supertags):
    """
    Return True if all clusters exist
    """
    cnt = Page.objects.filter(supertag__in=supertags, org=get_org()).aggregate(Count('id'))
    if cnt['id__count'] != len(supertags):
        return False
    return True


def names_conflict(clusters, with_children):
    """Check if some of destination clusters already exist

    Return list of conflicting page names including children,
    otherwise empty one

    filter out redirects - they are not moved"""
    conflicts = []
    for source, result in clusters.items():
        # parent len
        length = len(source)
        full_source = [source]
        if with_children:
            full_source.extend(
                Page.objects.filter(
                    supertag__startswith=source + '/',
                    redirects_to__isnull=True,
                    org=get_org(),
                ).values_list('supertag', flat=True)
            )

        # result len
        destination_tags = []
        for st in full_source:
            part = st[length:].strip('/')
            if part:
                part = '/' + part
            destination_tags.append(result + part)
        conflicts = conflicts + list(Page.objects.filter(supertag__in=destination_tags, org=get_org()))
    return conflicts


def clusters_accessible(supertags, user):
    """
    Return list of clusters which are not accessible due to access
    restrictions
    """
    access_forbidden = []
    for supertag in supertags:
        if get_access_status(supertag, user) < ACCESS_COMMON:
            access_forbidden.append(supertag)
    return access_forbidden


def result_accessible(clusters, user):
    """
    Return True if parent makes resulting cluster not accessible to user

    assume supertags are on one level
    """
    check_destinations = []
    pages = Page.objects.filter(supertag__in=list(clusters.keys()), org=get_org()).prefetch_related('authors')
    authors_map = {page.supertag: page.get_authors() for page in pages}
    for source, dest in clusters.items():
        if user in authors_map.get(source):
            continue
        check_destinations.append(dest)

    cluster_access = []
    for supertag in check_destinations:
        dest = '/'.join(supertag.split('/')[:-1])
        access = get_access_status(dest, user)
        if access == ACCESS_DENIED:
            cluster_access.append(dest)
    if cluster_access:
        return cluster_access
    return None


def check_into_itself(clusters):
    conflicts = {}
    for source, result in clusters.items():
        if source == result or result.startswith(source + '/'):
            conflicts[source] = result
    return conflicts


def check_not_redirect(supertags):
    return [
        p.supertag
        for p in Page.objects.filter(Q(redirects_to__isnull=False) & Q(supertag__in=supertags) & Q(org=get_org()))
    ]


def check_allowed_supertags(clusters):
    corrections = {}
    for st, dest in clusters.items():
        corrections[st] = translit(dest)
    return corrections


class Cluster(object):
    offset = None

    def __init__(self, login_or_user):
        if not login_or_user:
            raise Exception('login must not be empty')
        if isinstance(login_or_user, User):
            self.username = login_or_user.username
            self.user = login_or_user
        else:
            self.username = login_or_user
            self.user = org_user().get(username=login_or_user)

    @transaction.atomic
    def move(self, tag_clusters, with_children=True, allow_reserved_pages=False):
        """
        Move pages (possibly with children) to a new location.
        Returns operation code (similar to http status codes, 200 - success) and
        relevant data.

        @type tag_clusters: dict
        @param tag_clusters: supertags map. Format {<old supertag0>: <new supertag0>, <old supertag1>: <new supertag1>}
        @param with_children: move subpages?
        @param allow_reserved_pages: allow reserved pages to be moved?
        @rtype: tuple
        @return: (status_code, operation data)
        """
        if not tag_clusters:
            return [412, M_NOTHING_TO_DO, None]

        for key, value in tag_clusters.items():
            if any(
                len(tag) > 250
                for tag in (
                    key,
                    value,
                )
            ):
                return [409, M_TOO_LONG_NAME, [value]]

        clusters = check_allowed_supertags(tag_clusters)

        source = list(clusters.keys())
        conflicts = check_same_parent(source)
        if conflicts:
            return [412, M_DIFFERENT_S, conflicts]

        destination = list(clusters.values())
        conflicts = check_same_parent(destination)
        if conflicts:
            return [412, M_DIFFERENT_R, conflicts]

        conflicts = check_into_itself(clusters)
        if conflicts:
            return [412, M_INTO_ITSELF, conflicts]

        exists = all_clusters_exist(source)
        if not exists:
            return [400, M_DOESNT_EXIST, []]

        redirects = check_not_redirect(source)
        if redirects:
            return [409, M_IS_REDIRECT, redirects]

        conflicts = names_conflict(clusters, with_children)
        if conflicts:
            show_conflicts = []
            for p in conflicts:
                if get_access_status(p.supertag, self.user) > ACCESS_DENIED:
                    show_conflicts.append(p.supertag)
            return [409, M_CANT_OVERWRITE, show_conflicts]

        if not allow_reserved_pages:
            conflicts = any_page_reserved(source, with_children)
            if conflicts:
                return 409, M_SOURCE_RESERVED, conflicts

            conflicts = destination_reserved(clusters, with_children)
            if conflicts:
                return 409, M_DESTINATION_RESERVED, conflicts

        conflicts = clusters_accessible(source, self.user)
        if conflicts:
            return [403, M_PERMISSION_RESTRICTIONS, conflicts]
        access_conflict = result_accessible(clusters, self.user)
        if access_conflict is not None:
            return [403, M_WILL_LOSE_ACCESS, access_conflict]
        try:
            with transaction.atomic():
                for source, dest in clusters.items():
                    self.do_move_cluster(
                        source_supertag=source,
                        to_supertag=dest,
                        new_tag=tag_clusters[source],
                        with_children=with_children,
                    )
        except IntegrityError as exc:
            if 'Data truncated for column' in str(exc):
                return [409, M_TOO_LONG_NAME, [dest]]
            logger.exception('Cannot move: "%s"', exc)
            return [500, M_UNKOWN, []]
        return [200, clusters]

    def do_move_cluster(self, source_supertag, to_supertag, new_tag=None, with_children=True):
        """
        Перенести кластер из source в dest

        Для каждой страницы из кластера перенос выглядит так:
          пусть переносим страницу id=1, tag='foo' по адресу /bar.
          1. UPDATE pages_page SET tag='bar' WHERE id=1;
             заменили тег и все остальное странице, сохранив id
             при этом страницы foo больше нет
             но все связанные ранее с foo сущности остались связаны с новой страницей bar
          2. INSERT INTO pages_page (tag, type) VALUES ('foo', 'REDIRECT')
             создали заново страницу 'foo', но теперь она - редирект на bar
             2.1 досоздаем все необходимые для существования страницы сущности:
                 ревизии, ссылки, редиректы, метаданные итп
        """
        now = timezone.now()
        source_base = Page.objects.get(supertag=source_supertag, org=get_org())
        s_tag_slash_leaps = len(source_base.tag.split('/'))
        s_supertag_l = len(source_base.supertag)
        try:
            if to_supertag.split('/')[:-1]:
                dest_base = Page.objects.get(supertag='/'.join(to_supertag.split('/')[:-1]), org=get_org())
                d_tag = '/'.join((dest_base.tag, new_tag.split('/')[-1]))
            else:
                d_tag = new_tag
        except Page.DoesNotExist:
            d_tag = '/'.join(to_supertag.split('/')[:-1] + new_tag.split('/')[-1:])

        access_list_before = Access.objects.filter(page=source_base)
        if access_list_before:
            acl = []
            for access in access_list_before:
                acl.append(
                    '[is_common=%s, is_owner=%s, is_anonymous=%s, staff=%s, group=%s]'
                    % (access.is_common, access.is_owner, access.is_anonymous, access.staff, access.group)
                )
            acl = '+'.join(acl)
        else:
            acl = 'undefined'
        debug_logger.info(
            'WIKI9394 moving page=%s to=%s with_children=%s access=%s', source_supertag, to_supertag, with_children, acl
        )

        all_pages_query_set = Page.active.select_for_update().filter(redirects_to__isnull=True, org=get_org())

        if with_children:
            all_pages_query_set = all_pages_query_set.filter(
                Q(supertag__startswith=source_supertag + '/') | Q(supertag=source_supertag)
            )
        else:
            all_pages_query_set = all_pages_query_set.filter(supertag=source_supertag)

        created_tags_in_wiki, created_supertags_in_wiki = [], []
        source_access = interpret_raw_access(get_raw_access(source_base))

        page_redirect_map = {}  # обратный маппинг "итоговая страница на новом месте" - "редирект"

        for page in queryset_iterator(all_pages_query_set, chunk_size=50):
            move_page_to_supertag = to_supertag + page.supertag[s_supertag_l:]
            result_tag = '/'.join([d_tag] + page.tag.split('/')[s_tag_slash_leaps:])

            # обрабатываем ссылки на вики в теле страницы
            if page.page_type != Page.TYPES.GRID:
                process_links(
                    page_to_move=page,
                    user=self.user,
                    dest_supertag=move_page_to_supertag,
                    top_cluster=source_supertag,
                    top_cluster_to=to_supertag,
                    with_children=with_children,
                )

            tag_before, supertag_before = page.tag, page.supertag

            # страница перемещается
            page.supertag = move_page_to_supertag
            page.tag = result_tag
            page.modified_at_for_index = now
            page.save()

            created_tags_in_wiki.append(page.tag)
            created_supertags_in_wiki.append(page.supertag)

            # на месте страницы создается редирект
            page_data = dict(
                supertag=supertag_before,
                tag=tag_before,
                org=get_org(),
                title=page.title,
                modified_at=now,
                modified_at_for_index=now,
                created_at=now,
                page_type=page.page_type,
                redirects_to=page,
                owner_id=self.user.id,
                last_author_id=self.user.id,
                mds_storage_id=page.mds_storage_id,
                formatter_version=page.formatter_version,
                actuality_status=ACTUALITY_STATUS.unspecified,
                is_documentation=page.is_documentation,
                depth=len(supertag_before.split('/')),
            )
            if settings.IS_INTRANET:
                page_data['is_official'] = page.is_official
            redirect_at_source_place = Page(**page_data)

            page_redirect_map[page] = redirect_at_source_place

        # создаем все редиректы
        Page.objects.bulk_create(list(page_redirect_map.values()))
        # bulk_create не знает id получившихся редиректов,
        # поэтому маппинг делаем заново за один запрос к базе
        redirects_with_ids = {
            redirect.supertag: redirect
            for redirect in page_dao.get_pages_by_supertags(
                [redirect.supertag for redirect in list(page_redirect_map.values())]
            )
        }
        for page, redirect in page_redirect_map.items():
            redirects_with_id = redirects_with_ids[redirect.supertag]
            redirects_with_id.authors.add(*page.authors.all())
            redirects_with_id.access_set.add(
                *[self._clone_access(access, redirects_with_id) for access in page.access_set.all()]
            )
            page_redirect_map[page] = redirects_with_id

        for old_page_id, new_page_id in page_redirect_map.items():
            # PageLink должен вести на страницу-редирект,
            # потому что у нее остались старые тег-супертег
            PageLink.objects.filter(from_page=old_page_id).update(from_page=new_page_id)
            PageLink.objects.filter(to_page=old_page_id).update(to_page=new_page_id)

        Revision.objects.bulk_create(
            [Revision.objects.produce_from_page(page) for page in list(page_redirect_map.values())]
        )

        acls_to_create = []
        if source_access['is_inherited'] or source_access['is_common_wiki']:
            if source_access['is_owner']:
                acls_to_create.append(dict(page=source_base, is_owner=True))
            elif source_access['is_common'] or source_access['is_common_wiki']:
                acls_to_create.append(dict(page=source_base, is_common=True))
            elif source_access['is_anonymous']:
                acls_to_create.append(dict(page=source_base, is_anonymous=True))
            else:
                for group in source_access['groups']:
                    acls_to_create.append(dict(page=source_base, group=group))
                for user in source_access['users']:
                    if isinstance(user, User):
                        acls_to_create.append(dict(page=source_base, user=user))
                    else:  # Staff
                        acls_to_create.append(dict(page=source_base, staff=user))
            if acls_to_create:
                if any(access_before.is_common for access_before in access_list_before):
                    if any(access_after.get('is_common', False) for access_after in acls_to_create):
                        logger.error(
                            'WIKI9394BUG adding is_common access for page %s which already has is_common access',
                            source_supertag,
                        )
                        raise ValueError('Invalid access after move')

                Access.objects.bulk_create([Access(**kwargs) for kwargs in acls_to_create])

                access_list_after = Access.objects.filter(page=source_base)
                if len([access_after for access_after in access_list_after if access_after.is_common]) > 1:
                    logger.error('WIKI9394BUG created > 1 is_common access while moving page %s', source_supertag)
                    raise ValueError('Invalid access after move')

            page_signals.access_changed.send(sender=self.__class__, page_list=[source_base, to_supertag])

        nearest_parent = hierarchy.get_nearest_existing_parent(to_supertag)
        if nearest_parent:
            pages = list(page_redirect_map)
            subscribers = [u for u in get_users_subscribed_to_page(nearest_parent) if not uses_new_subscriptions(u)]
            subscription.subscribe_to(pages, *subscribers, subscribe_to_cluster=True)

        # Написать логику обновления подписок. Если переносим /foo/bar в /xxx/bar,
        # при этом пользователь Alice подписана на xxx как на кластер, а на /foo/bar/yyy как на страницу,
        # то Alice должна стать подписана на /foo/bar/yyy как на кластер

        AbsentPage.objects.filter(
            Q(to_supertag__in=created_supertags_in_wiki, from_page__org=get_org())
            | Q(to_tag__in=created_tags_in_wiki, from_page__org=get_org())
        ).delete()

    def attach(self, page, supertag, root, user, children_count, access=None):
        params = {'url': str('/' + supertag)}
        if page:
            if access == ACCESS_DENIED:
                params['lock'] = '1'
                params['name'] = ''  # AAA! this is how arkel@ wants
            else:
                if access == ACCESS_RESTRICTED:
                    params['key'] = '1'
                if page.has_redirect():
                    params['redirect'] = '1'
                    #                params['id']=str(page.pk)
                params['name'] = str(page.title)
        if children_count > 0:
            params['children'] = str(children_count)
        e = etree.Element('page', **params)
        root.append(e)
        return e

    def list(self, cluster_name, depth=None, limit=3000):
        query_cluster_name = translit(cluster_name)

        qs = Page.active.filter(supertag__startswith=query_cluster_name, org=get_org())

        if depth is not None:
            # depth starts from 0, models' depth from one
            qs = qs.filter(depth=depth + 1)
            # qs = qs.extra(
            #     where=[" length(supertag) - length(replace(supertag, '/', '')) = %s "],
            #     params=[str(depth)],
            # )

        return qs.order_by('supertag')[:limit]

    def _clone_access(self, acl, page):
        new_acl = Access(
            page=page,
            staff=acl.staff,
            group=acl.group,
            is_common=acl.is_common,
            is_owner=acl.is_owner,
            is_anonymous=acl.is_anonymous,
        )
        new_acl.save()

        return new_acl


def filter_public_page(cluster_name, access_list):
    def _filter(page):
        access = access_list.get(page.supertag, None)
        if page.supertag == cluster_name:
            return True
        if access is None or access == ACCESS_DENIED:
            return False
        return True

    return _filter


class MessagesMoveDict(dict):
    def get(self, key, default=None):
        value_or_func = super(MessagesMoveDict, self).get(key, default)
        if callable(value_or_func):
            return value_or_func()
        else:
            return value_or_func


M_TRUNK = 'trunk'
M_INTO_ITSELF = 'into_itself'
M_DIFFERENT_S = 'different_source_parents'
M_DIFFERENT_R = 'different_result_parents'
M_DOESNT_EXIST = 'cluster_doesnt_exist'
M_IS_REDIRECT = 'cluster_is_redirect'
M_NOTHING_TO_DO = 'nothing_to_do'
M_CANT_OVERWRITE = 'cant_overwrite'
M_PERMISSION_RESTRICTIONS = 'permission_restrictions'
M_WILL_LOSE_ACCESS = 'will_lose_access'
M_SUCCESS = 'success'
M_INCORRECT_SUPERTAG = 'incorrect_supertag'
M_UNKOWN = 'unknown_error'
M_CLUSTER_COLLISION = 'cluster_collision'
M_INCORRECT_SUPERTAG_FORMAT = 'incorrect_format'
M_DESTINATION_DOES_NOT_EXIST = 'route66'
M_HOME_PAGE_FORBIDDEN = 'home_page_forbidden'
M_TOO_LONG_NAME = 'name_is_too_long'
M_SOURCE_RESERVED = 'source_has_reserved_cluster'
M_DESTINATION_RESERVED = 'destination_has_reserved_cluster'
MESSAGES_MOVE = MessagesMoveDict(
    {
        M_TRUNK: ugettext_lazy('You may view top level clusters only by using address bar'),
        M_INTO_ITSELF: ugettext_lazy('You are trying to move these cluster(s) into themselves'),
        M_DIFFERENT_S: ugettext_lazy('Clusters moved at a time must have the same parent'),
        M_DIFFERENT_R: ugettext_lazy('You can\'t move clusters to different parents at a time'),
        M_DOESNT_EXIST: ugettext_lazy('Some clusters you are trying to move, do not exist'),
        M_IS_REDIRECT: ugettext_lazy('One of clusters is a redirect'),
        M_NOTHING_TO_DO: ugettext_lazy('Didn\'t receive list of clusters to move'),
        M_CANT_OVERWRITE: ugettext_lazy('moveCannotOverwrite'),
        M_PERMISSION_RESTRICTIONS: ugettext_lazy('You don\'t have enough permissions to move this clusters'),
        M_WILL_LOSE_ACCESS: ugettext_lazy(
            'Operation not permitted, because when you move cluster there it will become not accessible to you'
        ),
        M_SUCCESS: ugettext_lazy('The transfer completed successfully'),
        M_INCORRECT_SUPERTAG: ugettext_lazy(
            'Name must start with alphanumeric character or minus and must consist of letters, digits, '
            'underscores, dots and spaces'
        ),
        M_UNKOWN: lambda: ugettext_lazy('Unhandled error, please report {email_contact}').format(
            email_contact=get_support_contact()
        ),
        M_CLUSTER_COLLISION: ugettext_lazy('Page with such address already exists in this cluster'),
        M_INCORRECT_SUPERTAG_FORMAT: ugettext_lazy('Please write name of page in correct format'),
        M_DESTINATION_DOES_NOT_EXIST: ugettext_lazy('You are trying to move clusters to nowhere'),
        M_HOME_PAGE_FORBIDDEN: ugettext_lazy('Moving HomePage cluster is a forbidden operation'),
        M_TOO_LONG_NAME: ugettext_lazy('Name for a cluster can not be longer than 250'),
        M_SOURCE_RESERVED: ugettext_lazy('move:CantMoveReservedCluster'),
        M_DESTINATION_RESERVED: ugettext_lazy('move:CantCreateReservedCluster'),
    }
)
