from wiki.org import get_org
from wiki.pages.models import Page, PageWatch
from wiki.sync.connect.base_organization import as_base_organization


def _filter_by_pages_and_user(pages, user):
    """
    Вернуть QuerySet с подписками пользователя user на pages.

    Избегайте использовать напрямую, используйте filter_user_page_watches.

    @type pages: list
    @param pages: может быть списком страниц или списком id-шников страниц.
    @type user: User
    @rtype: QuerySet
    """
    return PageWatch.objects.filter(page__in=pages, user=user.username).select_related('page')


def _filter_by_supertags_and_user(supertags, user):
    """
    Вернуть QuerySet с подписками пользователя user на supertags.

    Избегайте использовать напрямую, используйте filter_user_page_watches.

    @type supertags: list
    @type user: User
    @rtype: QuerySet
    """
    return PageWatch.objects.filter(
        page__supertag__in=supertags, page__org=get_org(), user=user.username
    ).select_related('page')


def filter_pages_watches(pages):
    """
    Вернуть подписки на заданные страницы в словаре, где ключ - Page, значение - list.

    @param pages: список из Page или int - id-шников страниц
    @type pages: list
    @rtype: dict
    """
    intermediate_result = PageWatch.objects.filter(page__in=pages)

    if pages and not isinstance(list(pages)[0], Page):
        pages = Page.objects.filter(pk__in=pages)

    pages_map = {page.id: page for page in pages}
    result = {page: [] for page in pages}
    for page_watch in intermediate_result:
        page = pages_map[page_watch.page_id]
        result[page].append(page_watch)
    return result


def get_users_subscribed_to_page(page, subscribed_to_cluster=None, sort=False):
    """
    Вернуть пользователей, которые подписаны на page.

    @type subscribed_to_cluster: bool
    @param subscribed_to_cluster: Если None, то вернуть все подписки, иначе отфильтровать по этому признаку.
    @rtype: QuerySet
    @return: список из User
    """
    page_watches = filter_pages_watches([page])[page]
    if subscribed_to_cluster is not None:
        page_watches = [pw for pw in page_watches if pw.is_cluster == subscribed_to_cluster]

    organization = as_base_organization(page.org)
    users = organization.get_users().filter(username__in=[pw.user for pw in page_watches])
    if sort:
        users = users.order_by(
            'last_name',
            'first_name',
            'username',
        )
    return users


def filter_user_page_watches(user, pages=None, supertags=None):
    """
    Вернуть множество подписок пользователя на заданные страницы.

    Для одного объекта используйте get_user_page_watch из logic.

    @type user: User
    @type pages: list[Page]
    @type supertags: list[str]
    @rtype: list[PageWatch]
    @return: list of PageWatch
    """
    result = set()
    if pages:
        result = set(_filter_by_pages_and_user(pages, user))
    if supertags:
        for page_watch in _filter_by_supertags_and_user(supertags, user):
            result.add(page_watch)
    return result


def create_subscriptions(page_watches: list[PageWatch], ignore_conflicts: bool = False) -> list[PageWatch]:
    """
    Создать подписки для переданных PageWatch

    Быстро, но не отсылает сигналов.
    Кажется PageWatch никто не слушает и это ок
    """
    return PageWatch.objects.bulk_create(page_watches, batch_size=50, ignore_conflicts=ignore_conflicts)
