
import logging
import operator
from itertools import groupby

from cached_property import cached_property
from django.conf import settings

from wiki.pages.dao import access as access_dao

logger = logging.getLogger(__name__)


def get_access(page):
    """
    Получить объект PageAccess с доступами для страницы `page`.
    """
    chain = list(page.ancestors) + [page]
    access_objects = access_dao.get_access_data(chain)
    access_objects_by_page_id = _group_access_objects(access_objects)

    page_access = None
    for page_ in chain:
        page_access = PageAccess(
            page=page_, access_objects=access_objects_by_page_id.get(page_.id, []), parent_access=page_access
        )
    return page_access


def _group_access_objects(access_objects):
    get_page_id = operator.attrgetter('page_id')
    access_objects = sorted(access_objects, key=get_page_id)
    return {
        page_id: list(page_access_objects) for page_id, page_access_objects in groupby(access_objects, key=get_page_id)
    }


def is_external_employee(user):
    try:
        return user.staff.is_external_employee
    except Exception:
        logger.exception('Unable to determine is_external_employee')
        return False


class PageAccess(object):
    def __init__(self, page, access_objects, parent_access):
        self.page = page
        self.access_objects = access_objects
        self.parent_access = parent_access

    def __repr__(self):
        return f'<{self.__class__.__name__}: {self.page.supertag} {self.type}>'

    @cached_property
    def effective_type(self):
        """
        Если inherited — то берем effective_type от родительского кластера
        """
        if self.is_inherited:
            return self.parent_access.effective_type
        return self.type

    @cached_property
    def type(self):
        """
        owner|restricted|common|anonymous|inherited
        """
        if self.is_inherited:
            return 'inherited'
        if self.is_anonymous:
            return 'anonymous'
        if self.is_owner:
            return 'owner'
        if self.is_common:
            return 'common'
        if self.is_restricted:
            return 'restricted'
        raise RuntimeError('Unknown access type for page %s' % self.page.supertag)

    @cached_property
    def has_parent(self):
        return self.parent_access is not None

    @cached_property
    def is_inherited(self):
        return self.has_parent and not self.access_objects

    @cached_property
    def is_anonymous(self):
        access_object = self._get_only_access_object()
        if access_object:
            return access_object.is_anonymous
        return False

    @cached_property
    def is_owner(self):
        access_object = self._get_only_access_object()
        if access_object:
            return access_object.is_owner
        return False

    @cached_property
    def is_common(self):
        """
        Кажется так:
        доступ common
        когда есть одна запись с is_common=True (неважно есть ли родитель)
        ИЛИ
        когда в базе нет записей для страниц без родителя
        """
        access_object = self._get_only_access_object()
        if access_object:
            return access_object.is_common
        if not self.has_parent:
            return not self.access_objects
        return False

    @cached_property
    def is_restricted(self):
        return any(obj.staff_id or obj.group_id for obj in self.access_objects)

    @cached_property
    def restrictions(self):
        if self.is_inherited:
            parent_restrictions = self.parent_access.restrictions
            if parent_restrictions:
                for author in self.parent_access.page.get_authors():
                    if author not in parent_restrictions['users']:
                        parent_restrictions['users'].append(author)
            return parent_restrictions

        if not self.is_restricted:
            return

        allowed_users = set()
        allowed_groups = set()
        for obj in self.access_objects:
            if obj.staff:
                allowed_users.add(obj.staff.user)
                continue
            if obj.group:
                if settings.IS_BUSINESS:
                    obj.group = obj.group.group
                allowed_groups.add(obj.group)
                continue

        page_restrictions = {
            'users': list(allowed_users),
            'groups': list(allowed_groups),
        }

        return page_restrictions

    @cached_property
    def external_users_restriction(self):
        if not self.restrictions:
            return None

        return [user for user in self.restrictions['users'] if is_external_employee(user)]

    @cached_property
    def external_groups_restriction(self):
        if not self.restrictions:
            return None

        return [group for group in self.restrictions['groups'] if group.externals_count > 0]

    @cached_property
    def is_opened_to_external(self):
        if not self.restrictions:
            return False

        return any(
            [
                self.external_users_restriction,
                self.external_groups_restriction,
            ]
        )

    def _get_only_access_object(self):
        if len(self.access_objects) == 1:
            return self.access_objects[0]
