from typing import Optional

from wiki.api_v2.public.pages.schemas import ActualitySchema, PageDetailsSchema
from wiki.notifications.models import PageEvent
from wiki.pages.dao.actuality import (
    get_actual_external_link_urls,
    get_actuality_mark,
    get_actuality_mark_external_links,
    get_actuality_mark_links,
    get_linked_actual_pages_tags,
)
from wiki.pages.dao.page import get_pages_by_supertags
from wiki.pages.logic.comment import add_comment
from wiki.pages.logic.tags import string_to_supertag
from wiki.pages.logic.wf import format_comment
from wiki.pages.models import ActualityMark, ActualityMarkExternalLink, ActualityMarkLink, Page
from wiki.pages.models.consts import ACTUALITY_STATUS, ACTUALITY_STATUS_MAP, ActualityStatus
from wiki.users.user_data_repository import USER_DATA_REPOSITORY
from wiki.utils.errors import ValidationError
from wiki.utils.timezone import now
from wiki.utils.url import is_valid_url, is_wiki_url
from pydantic import HttpUrl


def get_actuality(page: Page) -> ActualitySchema:
    status = ACTUALITY_STATUS_MAP[page.realtime_actuality_status]
    a = ActualitySchema(status=status)

    if not page.has_manual_actuality_mark:
        return a

    mark = get_actuality_mark(page.id)
    if mark is None:
        return a

    a.comment = None
    if mark.comment and mark.comment.body:
        a.comment = mark.comment.body

    a.marked_at = page.actuality_marked_at
    a.user = (USER_DATA_REPOSITORY.orm_to_user_schema(mark.user),)

    if status == ActualityStatus.OBSOLETE:
        a.actual_pages = [
            PageDetailsSchema.serialize(link.actual_page) for link in ActualityMarkLink.objects.filter(page=page)
        ]
        a.external_links = list(get_actual_external_link_urls(page.id))

    return a


class ActualityViewDetails(object):
    """
    Все необходимые для отображения детали отметки об (не)актуальности страницы.
    """

    def __init__(self, mark, comment_bemjson, links):
        """
        @type mark: ActualityMark
        @type comment_bemjson: dict
        @type links: list

        @param links: список относительных путей страниц (слеш + тэг) (basestring)
        """
        self.mark = mark
        self.comment_bemjson = comment_bemjson
        self.links = links


def get_page_actuality_details(page, request):
    """
    @type page: Page
    @rtype: ActualityViewDetails
    """
    if not page.has_manual_actuality_mark:
        return None

    mark = get_actuality_mark(page.id)
    tags = get_linked_actual_pages_tags(page.id)
    tag_links = ['/' + x for x in tags]
    external_links = get_actual_external_link_urls(page.id)
    links = tag_links + list(external_links)

    if mark.comment is None:
        comment_bemjson = None
    else:
        comment_bemjson = format_comment(mark.comment)

    return ActualityViewDetails(mark, comment_bemjson, links)


def mark_page_actuality(
    user,
    page,
    is_actual,
    comment,
    mixed_links=None,
    external_links: Optional[HttpUrl] = None,
    actual_pages: Optional[Page] = None,
):
    """
    Помечает страницу актульной/устаревшей.
    Если среди указанных ссылок есть невалидная (не ведущая на страницу в вики), выбрасывает
    wiki.utils.errors.ValidationError со ссылкой в invalid_value.

    @param is_actual: True – актуальная, False – устаревшая
    @param comment: строка, может быть None

    @param mixed_links: список тэгов или УРЛов страниц (строки) ЛИБО external_links + pages

    @return None

    @type user: User
    @type page: Page
    @type is_actual: bool
    @type comment: basestring|None
    """
    if page.has_manual_actuality_mark:
        actuality_mark = get_actuality_mark(page.id)
        if actuality_mark:
            actuality_mark.delete()
        for actuality_mark_link in get_actuality_mark_links(page.id):
            actuality_mark_link.delete()
        for actuality_mark_external_link in get_actuality_mark_external_links(page.id):
            actuality_mark_external_link.delete()

    if external_links or actual_pages:
        _handle_external_links_and_pages(external_links, actual_pages, page)
    elif mixed_links:
        _handle_mixed_links(mixed_links, page)

    if is_actual:
        assert not mixed_links and not external_links and not actual_pages
        page.actuality_status = ACTUALITY_STATUS.actual
    else:
        page.actuality_status = ACTUALITY_STATUS.obsolete

    page.actuality_marked_at = now()
    page.save()

    if comment:
        comment = add_comment(user, page, comment, None)

    ActualityMark.objects.create(user=user, page=page, comment=comment)

    event = PageEvent(
        page=page,
        author=user,
        timeout=now(),
        event_type=PageEvent.EVENT_TYPES.mark_actual if is_actual else PageEvent.EVENT_TYPES.mark_obsolete,
    )
    if comment:
        event.meta = {'comment_id': comment.id}
    else:
        event.meta = {}
    event.save()


def _handle_external_links_and_pages(external_links, actual_pages, page):
    for actual_page in actual_pages:
        ActualityMarkLink.objects.create(page=page, actual_page=actual_page)
    for link in external_links:
        ActualityMarkExternalLink.objects.create(page=page, url=link)


def _handle_mixed_links(mixed_links, page):
    # Из списка переданных линков выбираем те, что ведут на внешние ресурсы
    external_links = [link for link in mixed_links if is_valid_url(link) and not is_wiki_url(link)]
    mixed_links = list(set(mixed_links) - set(external_links))
    actual_pages_supertags = list(map(string_to_supertag, mixed_links))
    actual_pages_supertags_unique = set(actual_pages_supertags)
    actual_pages = get_pages_by_supertags(actual_pages_supertags_unique)
    if len(actual_pages_supertags_unique) != len(actual_pages):
        found_pages_supertags = [_page.supertag for _page in actual_pages]
        links_iter = mixed_links.__iter__()
        for supertag in actual_pages_supertags:
            link = next(links_iter)
            if supertag not in found_pages_supertags:
                raise ValidationError(link)
    for actual_page in actual_pages:
        ActualityMarkLink.objects.create(page=page, actual_page=actual_page)
    for link in external_links:
        ActualityMarkExternalLink.objects.create(page=page, url=link)
