from datetime import timedelta

from django.db.models import Count

from wiki.notifications.generators.base import EventTypes
from wiki.notifications.models import PageEvent
from wiki.pages import signals
from wiki.pages.logic.subscription import subscribe_to
from wiki.utils import timezone


def change_authors_for_page(page, user, authors):
    previous_authors = page.authors.all()
    add_authors, remove_authors = remove_duplicated_authors(authors, previous_authors)
    _replace_authors([page], add_authors, remove_authors)
    create_change_authors_event(page, user, add_authors, remove_authors, with_subpages=False, pages_count=1)


def change_authors_for_cluster(root, user, authors, replace_authors=None):
    replace_authors = replace_authors or root.authors.all()

    # выбираем подстраницы, в соавторах которых есть все авторы из списка replace_authors
    cluster_pages_with_authors = (
        root.descendants.only('id', 'authors')
        .filter(authors__in=replace_authors)
        .annotate(c=Count('authors'))
        .filter(c=len(replace_authors))
    )

    from wiki.pages import access

    is_admin = access.is_admin(user)
    if not is_admin:
        # если пользователь не админ, то выбираем только те страницы, где у него есть права менять авторов
        cluster_pages_with_authors = cluster_pages_with_authors.filter(authors=user)

    # добавляем текущую страницу из запроса, если у пользователя есть права менять для нее авторов
    if set(replace_authors).issubset(set(root.get_authors())) and (is_admin or user in root.get_authors()):
        cluster_pages_with_authors = [root] + list(cluster_pages_with_authors)

    if cluster_pages_with_authors:
        add_authors, remove_authors = remove_duplicated_authors(authors, replace_authors)
        _replace_authors(list(cluster_pages_with_authors), add_authors, remove_authors)

        with_subpages = len(cluster_pages_with_authors) > 1
        pages_count = len(cluster_pages_with_authors)
        meta = {
            'added_authors': [author.username for author in add_authors],
            'removed_authors': [author.username for author in remove_authors],
            'with_subpages': with_subpages,
            'pages_count': pages_count,
        }
        PageEvent.objects.bulk_create(
            [
                PageEvent(
                    page=page,
                    author=user,
                    timeout=timezone.now() + timedelta(minutes=5),
                    event_type=EventTypes.change_authors,
                    notify=page.id == root.id,
                    meta=meta,
                )
                for page in cluster_pages_with_authors
            ]
        )


def can_change_authors(page, user):
    from wiki.pages import access

    return access.is_admin(user) or user in page.get_authors()


def remove_duplicated_authors(new_authors, previous_authors):
    """
    Удалить из переданных списков общих (повторяющихся в обоих списках) авторов
    """
    add_authors = list(set(new_authors).difference(set(previous_authors)))
    remove_authors = list(set(previous_authors).difference(set(new_authors)))
    return add_authors, remove_authors


def _replace_authors(pages, add_authors, replace_authors):
    for page in pages:
        page.authors.remove(*replace_authors)
        page.authors.add(*add_authors)

        if page.owner in replace_authors:
            page.owner = page.authors.order_by('staff__is_dismissed', 'id').select_related('staff').first()
            page.save()

    if pages:
        subscribe_to(pages, *add_authors)
        signals.access_changed.send(sender=None, page_list=pages)


def add_user_to_authors(page, user, new_author):
    page.authors.add(new_author)

    subscribe_to([page], new_author)

    create_add_author_event(page, user, new_author)
    signals.access_changed.send(sender=None, page_list=[page])


def create_change_authors_event(page, user, added_authors, removed_authors, with_subpages, pages_count, timeout=5):
    PageEvent(
        page=page,
        author=user,
        timeout=timezone.now() + timedelta(minutes=timeout),
        event_type=EventTypes.change_authors,
        notify=True,
        meta={
            'added_authors': [author.username for author in added_authors],
            'removed_authors': [author.username for author in removed_authors],
            'with_subpages': with_subpages,
            'pages_count': pages_count,
        },
    ).save()


def create_add_author_event(page, user, new_author, timeout=5):
    PageEvent(
        page=page,
        author=user,
        timeout=timezone.now() + timedelta(minutes=timeout),
        event_type=EventTypes.add_author,
        notify=True,
        meta={
            'new_author': new_author.username,
        },
    ).save()
