from typing import Optional, List, Set
import logging

from django.conf import settings
from django.contrib.auth import get_user_model

from wiki.cloudsearch.staff_client import STAFF_API_CLIENT
from wiki.pages.logic.authors import add_user_to_authors
from wiki.pages.models import Page
from wiki.users.models import User
from wiki.utils.wiki_robot import get_wiki_robot

logger = logging.getLogger(__name__)

BIG_CHIEFS_LOGINS = frozenset(['volozh', 'tigran', 'bunina', 'ivanov'])


def get_first_user(usernames: Set[str]) -> Optional[User]:
    if not usernames:
        return None
    query = (
        get_user_model()
        .objects.filter(username__in=usernames, staff__is_dismissed=False, staff__is_robot=False)
        .exclude(username__in=BIG_CHIEFS_LOGINS)
    )
    return query.first()


def get_alive_author(authors) -> Optional[User]:
    return (
        authors.filter(staff__is_dismissed=False, staff__is_robot=False).exclude(username__in=BIG_CHIEFS_LOGINS).first()
    )


def found_chief_by_usernames(usernames: List[str]) -> Optional[User]:
    found_user = None
    while usernames and not found_user:
        usernames = STAFF_API_CLIENT.get_chief(login=','.join(usernames))
        found_user = get_first_user(usernames=usernames)
    return found_user


def found_chief_or_author_alive(page: Page) -> Optional[User]:
    chief = found_chief_by_usernames(usernames=page.get_authors().values_list('username', flat=True))

    parent = page.nearest_parent
    while not chief and parent:
        chief = get_alive_author(authors=parent.get_authors()) or found_chief_by_usernames(
            usernames=parent.get_authors().values_list('username', flat=True)
        )
        parent = parent.nearest_parent
    return chief


def add_first_found_chief_to_authors(page: Page):
    if not settings.IS_INTRANET:
        return

    chief = found_chief_or_author_alive(page=page)
    if chief:
        add_user_to_authors(page, user=get_wiki_robot(), new_author=chief)
        logger.info(f'Added author "{chief.username}" to page "{page.supertag}" without live authors')
