import emoji
import logging
import re

from wiki.org import get_org
from wiki.pages.api import save_page
from wiki.pages.models import AbsentPage, Page, PageLink
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.sync.connect.get_organization_by_inner_id import get_organization_by_inner_id
from wiki.utils.dict_keypath import value_by_path
from wiki.utils.link_to_normalized_supertag import link_to_normalized_supertag
from wiki.utils.url import is_valid_url, is_wiki_url
from wiki.utils.wfaas.client import get_wfaas_client
from wiki.utils.wfaas.ast import NodeType, action_link
from wiki.utils.wfaas.errors import WfaasApiRequestError
from wiki.utils.wiki_robot import get_wiki_robot

logger = logging.getLogger(__name__)

WFAAS_SETTINGS = {
    'remark': {
        'woofmd': {'actions': [{'name': 'include'}, {'name': 'grid'}]},
    }
}

INCLUDE_PARAMS_RANK = {
    param: i
    for i, param in enumerate(
        [
            'page',
            'href',
            'if_office',
            'if_city',
            'if_country',
            'if_group',
            'if_lang',
            'notitle',
            'notoc',
            'nowarning',
            'from',
            'to',
            'expired',
            'login',
            'passwd',
            'formatter',
            'enc',
            # grid only params
            'width',
            'readonly',
            'num',
            'sort',
            'filter',
            'columns',
        ]
    )
}

notpage_link_template = re.compile('^(file|mailto):')


def page_links(page):
    page_body = ''
    if page.page_type == Page.TYPES.GRID:
        # Нам не важно, в какой ячейке находятся ссылки,
        # выбираем все текстовые ячейки, объединяем их вместе
        # и выбираем ссылки из всего блока
        text_fields = list(page.column_names_by_type('string'))
        cell_values = []
        for row in page.access_data:
            if row:
                for field_name in text_fields:
                    cell_value = row.get(field_name)
                    if not cell_value or not cell_value.get('raw'):
                        continue
                    cell_values.append(cell_value['raw'])

        page_body = '\n'.join(cell_values)

    elif page.page_type == Page.TYPES.PAGE:
        page_body = page.body
    else:
        return []

    ast = get_wfaas_client().raw_to_ast(page_body, WFAAS_SETTINGS)
    return ast.wiki_links()


def track_links(page, is_new_page):
    """
    Detect links to other wiki pages on "page" and write it into
    wiki.pages.models.PageLink.

    Работает с объектами как Page, так и Grid
    """
    supertags = set(link_to_normalized_supertag(link, page.supertag) for link in page_links(page))
    existent = dict(
        (p[1], p[0]) for p in Page.active.values_list('id', 'supertag').filter(supertag__in=supertags, org=get_org())
    )
    logger.info(f'[track links] page {page.id} has links to pages {existent}')
    absent = dict((ap[1], ap[0]) for ap in AbsentPage.objects.values_list('id', 'to_supertag').filter(from_page=page))
    linked = dict(
        (pl[1], pl[0]) for pl in PageLink.objects.values_list('id', 'to_page__supertag').filter(from_page=page)
    )

    to_create = {'link': [], 'absent': []}
    for stag in supertags:
        # Already linked, so don't do anything
        if stag in linked:
            del linked[stag]
        # Page exists and must be linked
        elif stag in existent:
            to_create['link'].append(existent[stag])
        # Page doesn't exist and already marked as absent
        elif stag in absent:
            del absent[stag]
        # Page doesn't exist and must be marked as absent
        else:
            to_create['absent'].append(stag)

    # Create links and absent pages
    if to_create['link']:
        PageLink.objects.bulk_create([PageLink(from_page=page, to_page_id=page_id) for page_id in to_create['link']])
    if to_create['absent']:
        AbsentPage.objects.bulk_create(
            [AbsentPage(from_page=page, to_supertag=stag) for stag in to_create['absent'] if len(stag) <= 250]
        )

    # Clean up
    if linked:
        PageLink.objects.filter(id__in=list(linked.values())).delete()
    if absent:
        AbsentPage.objects.filter(id__in=list(absent.values())).delete()

    if is_new_page:
        absent_pages = AbsentPage.objects.filter(to_supertag=page.supertag, from_page__org=get_org())

        PageLink.objects.bulk_create(
            [
                PageLink(from_page_id=page_id, to_page=page)
                for page_id in absent_pages.values_list('from_page__id', flat=True)
            ]
        )

        AbsentPage.objects.filter(to_supertag=page.supertag, from_page__org=get_org()).delete()


def make_link_markup(node, url):
    link_text = '(('
    link_text += url

    children = node.get('children', [])
    for child_node in children:
        node_type = child_node.get('type', '')
        node_value = child_node.get('value', None)
        if node_type == 'text' and node_value is not None:
            link_text += f' {node_value}'
            break

    link_text += '))'
    return link_text


def make_include_markup(node, url):
    node_name = node.get('name').lower()
    assert node_name in {'include', 'grid'}
    action_text = '{{%s ' % node_name
    params = []
    for param_name in sorted(node.get('params', []), key=lambda x: INCLUDE_PARAMS_RANK.get(x, 99)):
        if param_name in {'page', 'href'}:
            params.append(f'{param_name}="{url}"')
            continue

        value = node['params'][param_name]
        if value is not None:
            params.append(f'{param_name}="{value}"')
        else:
            params.append(param_name)

    action_text += ' '.join(params)
    action_text += '}}'
    return action_text


def rewrite_links(affected_pages, org_id: int | None):
    previous_to_new_slugs = {page.previous_slug: page.slug for page in affected_pages}
    new_to_previous_slugs = {page.slug: page.previous_slug for page in affected_pages}
    page_links = PageLink.objects.filter(to_page__in=[page.id for page in affected_pages]).distinct('from_page')

    organization: BaseOrganization = get_organization_by_inner_id(org_id)
    wiki_robot = get_wiki_robot(organization)

    wfaas_client = get_wfaas_client()
    node_types = {NodeType.ACTION, NodeType.WOMLINK, NodeType.LINK}
    markup_func = {
        NodeType.ACTION: make_include_markup,
        NodeType.WOMLINK: make_link_markup,
        NodeType.LINK: make_link_markup,
    }
    for page_link in page_links:
        page = page_link.from_page
        if page.page_type != Page.TYPES.PAGE:
            # Обрабатываем только обычные страницы.
            # Для гридов нужно получать AST для каждой ячейки,
            # на больших гридах это займет непозволительно много
            # времени и потенциально может убить wfaas.
            continue

        changed = False
        body = page.body

        if any([emoji.is_emoji(c) for c in body]):
            # В python и js длина эмодзи считается по разному
            # в результате, при наличии эмодзи в теле страницы,
            # значение offset в ast смещается и становится непригодным
            # для использования в python.
            continue

        try:
            ast = wfaas_client.raw_to_ast(body, WFAAS_SETTINGS)
        except WfaasApiRequestError as e:
            logger.error(f'[rewrite_links] failed to get ast for page {page}, error {e}')
            continue

        for node in reversed(list(ast.filter(node_types))):
            node_type = node.get('type')
            node_name = node.get('name')
            if node_type == NodeType.ACTION and node_name in {'include', 'grid'}:
                node_url = action_link(node) or ''
            else:
                node_url = node.get('url', '')

            is_external_link = is_valid_url(node_url) and not is_wiki_url(node_url)
            if not node_url or is_external_link or notpage_link_template.match(node_url):
                # Пустая ссылка или Внешняя ссылка или ссылка не на страницу (email, файл)
                continue

            # Получаем supertag страницы по node_url, в случае относительного линка, раскрываем
            # его относительно супертега текущей страницы, который был до переноса.
            # Проверяем был полученный supertag среди перемещенных страниц, и если был, переписываем его.
            supertag = link_to_normalized_supertag(node_url, new_to_previous_slugs.get(page.supertag, page.supertag))
            if supertag in previous_to_new_slugs:
                pos_start = value_by_path('position.start.offset', node)
                pos_end = value_by_path('position.end.offset', node)
                node_url = previous_to_new_slugs[supertag]
                if not node_url.startswith('/'):
                    node_url = '/' + node_url

                body = body[:pos_start] + markup_func[node_type](node, node_url) + body[pos_end:]
                changed = True

        if changed:
            save_page(page, body, user=wiki_robot)
