from typing import Union, List, Dict, Any

from wiki.legacy.url import Url
from wiki.pages.logic.etalon import get_etalon_page

from wiki.api_v2.public.pages.schemas import BreadcrumbSchema
from wiki.pages.models import Page

from django.conf import settings
from wiki.sync.connect.base_organization import BaseOrganization, as_base_organization


def missing_serializer_old(slug):
    name = slug.split('/')[-1]
    return {'tag': slug, 'url': '/' + slug, 'name': name, 'title': name, 'is_active': False}


def missing_serializer_new(slug):
    name = slug.split('/')[-1]
    return BreadcrumbSchema(title=name, slug=slug, page_exists=False)


def page_serializer_old(page):
    if isinstance(page, Page):
        return {
            'tag': page.tag,
            'url': page.url,
            'name': page.tag.split('/')[-1],
            'title': page.title,
            'is_active': True,
        }
    else:
        return {
            'tag': page['tag'],
            'url': '/' + page['supertag'],
            'name': page['tag'].split('/')[-1],
            'title': page['title'],
            'is_active': True,
        }


def page_serializer_new(page):
    if isinstance(page, Page):
        return BreadcrumbSchema(id=page.id, title=page.title, slug=page.slug, page_exists=True)
    else:
        return BreadcrumbSchema(id=page['id'], title=page['title'], slug=page['supertag'], page_exists=True)


def get_breadcrumbs_compat(page) -> Union[List[BreadcrumbSchema], List[Dict[str, Any]]]:
    """
    Возвращает информацию о родителях в порядке от наиболее близкого к корню в виде массива вида:
    [
        {
            'name': '...',  # Кусок тега
            'tag': '...',  # Тег
            'url': '...',  # относительный УРЛ страницы
            'title': '...'  # Заголовок
            'is_active': false # Флаг наличия страницы
        },
    ]
    """
    return _get_breadcrumbs(page=page, org=page.org, as_schema=False)


def get_breadcrumbs_from_page(page) -> Union[List[BreadcrumbSchema], List[Dict[str, Any]]]:
    return _get_breadcrumbs(page=page, org=page.org, as_schema=True)


def get_breadcrumbs_from_slug(slug, org: BaseOrganization) -> Union[List[BreadcrumbSchema], List[Dict[str, Any]]]:
    return _get_breadcrumbs(slug=slug, org=org.as_django_model(), as_schema=True)


def _get_breadcrumbs(org, as_schema=True, page=None, slug=None) -> Union[List[BreadcrumbSchema], List[Dict[str, Any]]]:
    if as_schema:
        page_serializer = page_serializer_new
        missing_serializer = missing_serializer_new
    else:
        page_serializer = page_serializer_old
        missing_serializer = missing_serializer_old

    if page:
        slug = page.slug

    supertags = [slug] + Url.get_ancestors(slug)
    supertags.reverse()

    if len(supertags) == 1 and page is not None:  # для ускорения доступа к страницам первого уровня
        return [page_serializer(page)]

    existing_pages = {}

    for ancestor in (
        Page.active.filter(supertag__in=supertags, org=org)
        .order_by('supertag')
        .values('id', 'supertag', 'title', 'tag')
    ):
        existing_pages[ancestor['supertag']] = ancestor

    organization = as_base_organization(org)

    missed_supertags = set(supertags) - set(existing_pages.keys())
    for missed_supertag in missed_supertags:
        if missed_supertag in settings.COPY_ON_WRITE_TAGS:
            etalon_page = get_etalon_page(organization, slug=missed_supertag)
            existing_pages[missed_supertag] = {
                'id': etalon_page.id,
                'supertag': etalon_page.supertag,
                'tag': etalon_page.tag,
                'title': etalon_page.title,
            }

    breadcrumbs = []

    for supertag in supertags:
        ancestor = existing_pages.get(supertag)

        if ancestor:
            breadcrumb = page_serializer(ancestor)
        else:
            breadcrumb = missing_serializer(supertag)

        breadcrumbs.append(breadcrumb)

    return breadcrumbs
