import copy
import logging

from wiki.api_v2.public.pages.page.sideeffects import apply_sideeffects_on_page_creation
from wiki.org import get_org
from wiki.pages import dao as pages_dao
from wiki.pages.logic import tags
from wiki.pages.models import Revision, Page
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.utils import timezone
from wiki.utils.db import on_commit
from wiki.utils.copy_on_write import is_copy_on_write
from wiki.utils.supertag import tag_to_supertag

log = logging.getLogger(__name__)


def clone_page(
    page: Page,
    new_tag: str,
    authors: list,
    last_author,
    organization: BaseOrganization = None,
    copy_files: bool = True,
    subscribe_me: bool = True,
    **kwargs
) -> Page:
    """
    Клонировать страницу со всеми атрибутами и прикрепленными к странице файлами.
    Копируемые значения атрибутов страницы можно перезаписать, передав новые значения в качестве аргументов функции
    в kwargs.
    У новой страницы будет только одна ревизия и одно событие об ее создании.
    """
    new_page = copy.copy(page)
    new_page.id = None
    # комментарии не копируем
    new_page.comments = 0

    new_page.tag = tags.clean_tag(new_tag)
    new_page.supertag = tag_to_supertag(new_page.tag)
    new_page.owner = authors[0]
    new_page.last_author = last_author
    new_page.created_at = new_page.modified_at = new_page.modified_at_for_index = timezone.now()
    if not is_copy_on_write(page):
        new_page.reference_page = page

    if organization:
        org = organization.as_django_model()
    else:
        org = get_org()
    new_page.org = org

    # переопределяем некоторые атрибуты значениями, переданными в параметрах метода
    for name, value in kwargs.items():
        setattr(new_page, name, value)

    new_page.save()

    new_page.authors.add(*authors)

    if copy_files:
        clone_files(page, new_page)
    else:
        new_page.files = 0
        new_page.save()

    revision = Revision.objects.create_from_page(new_page)
    on_commit(lambda: apply_sideeffects_on_page_creation(new_page, revision, subscribe_me=subscribe_me, is_silent=True))
    return new_page


def clone_files(source_page, target_page):
    """
    Скопировать файлы, прикрепленные к странице source_page в страницу target_page.
    """
    if source_page.files < 1:
        return

    file_set = pages_dao.get_files(source_page)
    now = timezone.now()
    for file in file_set:
        clone_file = copy.copy(file)
        clone_file.id = None
        clone_file.page = target_page
        clone_file.user = target_page.get_authors().first()
        clone_file.created_at = clone_file.modified_at = now
        clone_file.save()

    if file_set.count() != target_page.files:
        target_page.files = file_set.count()
        target_page.save()
