from wiki.api_v2.public.pages.schemas import CommentSchema
from wiki.notifications import logic as notify_logic
from wiki.pages.models import Comment, Page
from wiki.pages.models.consts import COMMENTS_STATUS
from wiki.pages.tasks.comments import UpdateCommentsStatusTask
from wiki.users.user_data_repository import USER_DATA_REPOSITORY
from wiki.utils.db import on_commit


def add_comment(user, page, body, parent_id=None):
    """
    Добавить новый комментарий к странице.

    @param user: пользователь, добавляющий комментарий
    @param body: строка с текстом комментария
    @param parent_id: id родительского комментария в дереве комментариев,
    может быть None

    @type user: User
    @type page: Page
    @type body: basestring
    @type parent_id: int|basestring|None
    @rtype: Comment
    """
    comment = Comment.objects.create(user=user, page=page, page_at=page.modified_at, body=body, parent_id=parent_id)
    notify_logic.create_page_event_add_comment(
        page=page,
        user=user,
        comment=comment,
    )
    return comment


def edit_comment(comment, user, body):
    comment.body = body
    comment.save()
    return comment


def delete_comment(comment, user):
    comment.status = False
    comment.save()


def change_comments_status(page, disabled, for_cluster):
    """
    Изменить статус доступности комментариев к странице или к целому кластеру
    """
    # мапа для пары булевых значений (disabled, for_cluster) и значений COMMENTS_STATUS
    comments_status_map = {
        (False, True): COMMENTS_STATUS.enabled,
        (False, False): COMMENTS_STATUS.enabled,
        (True, False): COMMENTS_STATUS.disabled_on_page,
        (True, True): COMMENTS_STATUS.disabled_on_cluster,
    }

    comments_status = comments_status_map[disabled, for_cluster]
    if page.comments_status != comments_status:
        page.comments_status = comments_status
        page.save()

    if for_cluster:
        child_pages = Page.active.get_descendants(page.supertag)
        child_pages_ids = child_pages.values_list('id', flat=True)
        if len(child_pages_ids) > 100:
            # если кластер большой, то выполняем обновление страниц в селери таске
            on_commit(
                lambda: UpdateCommentsStatusTask().apply_async(
                    tuple(),
                    dict(root_page_id=page.id, child_pages_ids=tuple(child_pages_ids), comments_status=comments_status),
                )
            )
        else:
            update_comments_status(child_pages_ids, comments_status)


def update_comments_status(pages_ids, comments_status):
    Page.active.filter(id__in=pages_ids).update(comments_status=comments_status)


def serialize_comment(orm_model: Comment) -> CommentSchema:
    return CommentSchema(
        id=orm_model.id,
        body=orm_model.body if orm_model.status else '',
        parent_id=orm_model.parent_id,
        author=USER_DATA_REPOSITORY.orm_to_user_schema(orm_model.user),
        created_at=orm_model.created_at,
        is_deleted=not orm_model.status,
    )
