import logging

from django_pgaas import atomic_retry
from pymongo import errors as pymongo_errors

from wiki.notifications import logic as notifications_logic
from wiki.org import get_org
from wiki.pages import api
from wiki.pages.api import create_event, save_page, subscribe_to_new_page, toggle_author_subscription
from wiki.pages.models import Page, Revision
from wiki.utils import timezone
from wiki.utils.backports.mds_compat import APIError
from wiki.utils.db import on_commit
from wiki.utils.supertag import tag_to_supertag
from wiki.utils.tasks import TrackLinksTask

log = logging.getLogger(__name__)


def create(tag, user, body, title, notify=True):
    title = title or tag
    supertag = tag_to_supertag(tag)

    page = Page(
        tag=tag,
        supertag=supertag,
        owner=user,
        title=title,
        last_author=user,
        modified_at=timezone.now(),
        org=get_org(),
        with_new_wf=True,
    )

    page.body = body
    page.save()
    page.authors.add(user)

    revision = Revision.objects.create_from_page(page)
    notifications_logic.create_page_event_create(page, user, revision=revision, notify=notify)

    api.subscribe_to_new_page(page, subscribe_author=True)

    _run_edit_side_effects(page, just_created=True)
    return page


def _run_edit_side_effects(page, just_created):
    from wiki.favorites_v2 import logic as favourites_logic
    from wiki.utils import tasks as utils_tasks

    try:
        on_commit(lambda: utils_tasks.TrackLinksTask().delay(page.id, is_grid=False, is_new_page=just_created))
        favourites_logic.page_changed(page, was_created=just_created)
    except pymongo_errors.PyMongoError:
        log.exception('Mongo error')


# TODO: WIKI-9737 - этот и метод выше делают очень похожие вещи, нужно отрефакторить
@atomic_retry()
def save_page_from_view(
    page, request, tag, title, body, section_id=None, authors=None, make_me_watcher=False, page_type=None
):
    """
    Сохранить страницу вики-форматтером.

    @type make_me_watcher: bool
    @param make_me_watcher: True - пользователь хочет быть подписанным на страницу, False - хочет быть отписан.
    """

    if page:
        try:
            page.body
        except APIError:
            # Не удалось прочитать запись, собираемся записать на ее место новую.
            # Это может быть источником ошибок при редактировании секции: не прочитали
            # страницу, но секцию пытаемся сохранить.
            # Рассчитываем на то, что хранилище всегда стабильно!
            # И надеемся, что этого не произойдет.
            log.exception('Serious problem with storage on read: storage_id:"%s"', page.mds_storage_id.name)
            page.mds_storage_id = None
            page.body = ''

    page, _, changed = save_page(
        page if page else tag,
        body,
        title=(title or None if page or title else tag),
        section_id=section_id,
        authors=authors,
        request=request,
        create_revision=False,
        page_type=page_type,
    )

    if not changed:
        return page

    request.tag = page.tag  # При сохранении страницы мог поменяться тег

    # Track links to other pages
    on_commit(lambda: TrackLinksTask().delay(page.id, is_grid=False, is_new_page=page.is_new_page))

    # Toggle subscription of revision's author.
    # Subscribe cluster's watchers if saved page is new.
    subscribe_author = make_me_watcher or page.is_new_page

    if page.is_new_page:
        subscribe_to_new_page(page, subscribe_author)
    else:
        toggle_author_subscription(page, subscribe_author)

    # Create the revision
    rev = Revision.objects.create_from_page(page)

    # Create log's event
    if request.POST.get('notify_immediately', request.GET.get('notify_immediately')):
        timeout = 0
    else:
        timeout = 20
    create_event(page.is_new_page, page, rev, timeout)

    return page
