from logging import getLogger

from django.conf import settings
from django.db import IntegrityError
from django.db.models import Q
from django.template.loader import render_to_string

from wiki.api_v2.exceptions import AlreadyExists
from wiki.pages.logic.clone import clone_page
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.sync.connect.models import Organization
from wiki.org import get_org, org_ctx
from wiki.pages.models import Page
from wiki.pages.models.copy_on_write_page import CopyOnWritePage
from wiki.utils.wiki_robot import get_wiki_robot

logger = getLogger(__name__)
"""

В эталонной организации есть эталонный кластер, который имеет вид:

etalon
etalon/(язык)
etalon/(язык)/homepage
etalon/(язык)/sandbox
etalon/(язык)/sandbox/*
etalon/(язык)/cluster_users
etalon/(язык)/personal_page
etalon/(язык)/personal_page/notes

Например:

etalon
etalon/ru
etalon/ru/homepage
etalon/ru/sandbox
etalon/ru/sandbox/markup
etalon/ru/sandbox/tables
etalon/ru/cluster_users
etalon/ru/personal_page
etalon/ru/personal_page/notes
etalon/en
etalon/en/homepage
etalon/en/sandbox
etalon/en/sandbox/markup
etalon/en/sandbox/tables
etalon/en/cluster_users
etalon/en/personal_page
etalon/en/personal_page/notes

При создании новой организации из эталонного кластера
копируются страницы homepage, cluster_users и кластер sandbox,
при этом берутся версии страниц, соответствующие языку организации.
При создании домашней страницы пользователя используются страницы personal_page и notes.

"""


def get_etalon_pages(clusters, lang):
    if not settings.ETALON_ORG_DIR_ID:
        logger.warn('ETALON_ORG_DIR_ID is not specified')
        return []

    etalon_org = Organization.objects.get(dir_id=settings.ETALON_ORG_DIR_ID)

    with org_ctx(etalon_org):
        qs = (
            Page.active.filter(org=get_org())
            .extra(where=["supertag LIKE 'etalon/__'"])
            .values_list('supertag', flat=True)
        )

        available_langs = {supertag[len('etalon/') :] for supertag in qs}
        lang = _get_lang(lang, available=available_langs)

        query = Q()

        for cluster in clusters:
            query |= Q(supertag__startswith='etalon/%s/%s' % (lang, cluster))

        pages = Page.active.filter(query, org=get_org()).order_by('supertag')

        return pages


def get_etalon_page(organization: BaseOrganization, slug: str = None, idx: int = None) -> Page | CopyOnWritePage:
    if idx and idx in settings.COPY_ON_WRITE_IDS:
        slug = settings.COPY_ON_WRITE_IDS[idx]

    if existing_page := organization.get_active_pages().filter(supertag=slug).first():
        return existing_page

    template_params = settings.COPY_ON_WRITE_TAGS[slug]
    available_languages = set(template_params) - {'id'}
    lang = _get_lang(lang=organization.get_lang(), available=available_languages)

    body_params = template_params[lang]

    id = template_params.get('id', None)
    title = body_params.get('title', slug)
    body = render_to_string(body_params.get('template', ''))

    page = CopyOnWritePage(id=id, supertag=slug, organization=organization, title=title, body=body)
    return page


def create_etalon_page(organization: BaseOrganization, slug: str = None, idx: int = None) -> Page:
    etalon_page = get_etalon_page(organization, slug=slug, idx=idx)
    if type(etalon_page) is Page:
        return etalon_page

    wiki_robot = get_wiki_robot(organization=organization)
    try:
        cloned_page = clone_page(
            page=etalon_page,
            new_tag=etalon_page.slug,
            authors=[wiki_robot],
            last_author=wiki_robot,
            organization=organization,
            subscribe_me=False,
            copy_files=False,
            # as kwargs
            is_autogenerated=True,
            _should_skip_reserved_protection=True,
            _skip_validate_version=True,  # теперь это реальная страница, версия которой отличается от дефолтной для cow
        )
    except IntegrityError:  # страница уже была создана
        raise AlreadyExists()

    return cloned_page


def convert_supertag(supertag):
    """Превращает etalon/(язык)/* в *"""
    clusters = supertag.split('/')
    return '/'.join(clusters[2:])


def _get_lang(lang: str, available: set[str]) -> str:
    if lang in available:
        return lang
    elif 'en' in available:
        return 'en'
    return sorted(available)[0]
