from datetime import timedelta
from django.conf import settings

from wiki.api_v2.public.upload_sessions.exceptions import InvalidUploadSession
from wiki.files.consts import ThumbnailFieldSchema, ThumbnailStatus
from wiki.files.models import File
from wiki.files.tasks import CreateThumbnailTask
from wiki.notifications.models import EventTypes, PageEvent
from wiki.pages.dao.files import get_user_css_for_intranet
from wiki.uploads.consts import UploadSessionTargetType, UploadSessionStatusType
from wiki.utils import timezone
from wiki.utils.db import on_commit


def get_user_css(page):
    """
    @return: None или путь до user.css
    @rtype: basestring
    """
    if settings.WIKI_CODE != 'wiki':
        return None
    return get_user_css_for_intranet(page)


def attach_file_from_upload_session(page, user, upload_session):
    if not upload_session.status == UploadSessionStatusType.FINISHED:
        raise InvalidUploadSession(debug_message=f'session {upload_session.session_id} should have status finished')

    if not upload_session.target == UploadSessionTargetType.ATTACHMENT:
        raise InvalidUploadSession(
            debug_message=f'session {upload_session.session_id} has wrong target type "{upload_session.target}"'
        )

    file_name = upload_session.file_name
    file_size = upload_session.file_size
    storage_key = upload_session.storage_key

    NOW = timezone.now()

    attached_file = File(
        page=page,
        user=user,
        name=file_name,
        url=File.get_unique_url(page, file_name),
        size=file_size,
        description='',
        created_at=NOW,
        modified_at=NOW,
        s3_storage_key=storage_key,
    )

    attached_file.save()

    page.files += 1
    page.save()

    upload_session.status = UploadSessionStatusType.USED
    upload_session.save()

    PageEvent(
        timeout=NOW + timedelta(minutes=20),
        page=page,
        author=user,
        meta={
            'filename': attached_file.name,
            'url': attached_file.url,
            'size': attached_file.size,
            'id': attached_file.id,
        },
        event_type=EventTypes.add_file,
        notify=True,
    ).save()

    # Для картинок нужно сгенерить thumbnail
    if File.is_image(attached_file.name):
        attached_file.thumbnail = ThumbnailFieldSchema(
            status=ThumbnailStatus.SCHEDULED,
            changed_at=NOW,
        ).serialize()
        attached_file.save()
        on_commit(lambda: CreateThumbnailTask().delay(file_id=attached_file.id))

    return attached_file
