import io
import os

from django.conf import settings
from django.utils.translation import ugettext_lazy as _

from wiki.api_v2.public.pages.exceptions import WrongFileExtension
from wiki.api_v2.public.upload_sessions.exceptions import InvalidUploadSession
from wiki.pages.logic.edit import save_page_from_view
from wiki.pages.logic.import_file.doc import to_wiki_text as to_wiki_text_from_docx
from wiki.pages.logic.import_file.import_errors import ImportException
from wiki.pages.logic.import_file.txt import to_wiki_text as to_wiki_text_from_txt
from wiki.uploads.consts import UploadSessionTargetType, UploadSessionStatusType

format_to_handler = {
    'docx': to_wiki_text_from_docx,
    'txt': to_wiki_text_from_txt,
}


def import_file(file, request):
    file_size_limit = settings.LIMIT__IMPORT_FILE_SIZE__BYTES
    if file_size_limit and file.size > file_size_limit:
        raise ImportException(
            'Maximum import file size {limit} Mb exceeded'.format(limit=int(file_size_limit / (1 << 20)))
        )

    file_extension = os.path.splitext(file.name)[1][1:].lower()
    if file_extension not in format_to_handler:
        raise ImportException(
            'Unsupported file extension: {unsupported_ext}, '
            'supported extensions: {supported_ext_list}'.format(
                unsupported_ext=file_extension, supported_ext_list=', '.join(list(format_to_handler.keys()))
            )
        )

    handler = format_to_handler[file_extension]
    wiki_text = handler(file, request.page, request.user)

    wiki_text_limit = settings.LIMIT__WIKI_TEXT_FOR_PAGE__BYTES
    if wiki_text_limit and len(wiki_text.encode('utf-8')) > wiki_text_limit:
        raise ImportException(_('Maximum page size {limit} Kb exceeded').format(limit=int(wiki_text_limit / (1 << 10))))

    return save_page_from_view(
        page=request.page,
        request=request,
        tag=request.page.tag,
        title=request.page.title,
        body=wiki_text,
    )


def import_file_from_upload_session(upload_session, page, user):
    if not upload_session.status == UploadSessionStatusType.FINISHED:
        raise InvalidUploadSession(debug_message=f'session {upload_session.session_id} should have status finished')

    if not upload_session.target == UploadSessionTargetType.IMPORT_PAGE:
        raise InvalidUploadSession(
            debug_message=f'session {upload_session.session_id} has wrong target type "{upload_session.target}"'
        )

    file_extension = os.path.splitext(upload_session.file_name)[1][1:].lower()
    if file_extension not in format_to_handler:
        supported_extensions = ', '.join(format_to_handler.keys())
        raise WrongFileExtension(
            details=f'Unsupported file extension: "{file_extension}", supported extensions: {supported_extensions}'
        )

    data_stream = upload_session.data_stream
    file_data = io.BytesIO(data_stream.read())

    handler = format_to_handler[file_extension]
    wiki_text = handler(file_data, page, user)
    page.body = wiki_text
    page.save()

    upload_session.status = UploadSessionStatusType.USED
    upload_session.save()
