
from io import BytesIO
from itertools import takewhile
from uuid import uuid4

from docx import Document
from docx.enum.shape import WD_INLINE_SHAPE
from docx.opc.constants import RELATIONSHIP_TYPE as RT
from docx.oxml.table import CT_Tbl
from docx.oxml.text.paragraph import CT_P
from docx.oxml.text.run import CT_R
from docx.shared import Inches
from docx.table import Table
from docx.text.paragraph import Paragraph
from docx.text.run import Run
from lxml import etree

from wiki.api_core.logic import files as files_logic
from wiki.utils.doc import get_closest_wiki_color
from wiki.utils.supertag import translit

_paragraph_style_to_wt = {
    'Heading 1': '==',
    'Heading 2': '===',
    'Heading 3': '====',
    'Heading 4': '=====',
    'Heading 5': '======',
    'Heading 6': '=======',
    'Heading 7': '========',
    'Heading 8': '=========',
    'Heading 9': '==========',
    'List Paragraph': '* ',
    'List Bullet': '* ',
    'List Number 3': '1. ',
}


def to_wiki_text(file, page, user):
    doc = Document(file)
    wt = _process_file(doc)
    _add_images(doc, page, user, wt)
    return ''.join(wt)


def _process_file(doc):
    rels = doc.part.rels
    hyperlinks = [r._target for r in list(rels.values()) if r.reltype == RT.HYPERLINK]
    doc_meta = _DocMeta(hyperlinks)

    wt = []

    if 'Table of Contents' in doc._body._body.xml:
        wt.append('{{toc}}\n')

    for element in _iterate_over_paragraphs_and_tables(doc):
        if isinstance(element, Paragraph):
            _append_wt_of_paragraph(element, wt, doc_meta)
        else:
            wt.append('#|\n')
            for row in element.rows:
                wt.append('||')
                cells = row.cells
                for i, cell in enumerate(cells):
                    for p in cell.paragraphs:
                        _append_wt_of_paragraph(p, wt, doc_meta)
                    if i != len(cells) - 1:
                        wt.append('|')
                wt.append('||\n')
            wt.append('|#\n')

    return wt


def _append_wt_of_paragraph(paragraph, wt, doc_meta):
    try:
        if paragraph.paragraph_format.left_indent and paragraph.paragraph_format.left_indent > 0:
            tab_count = int(paragraph.paragraph_format.left_indent / Inches(0.5))
            wt.append(tab_count * '  ')
    except ValueError:
        # фикс WIKI-11576 (ValueError: invalid literal for int() with base 10: '566.9291338582675')
        wt.append('  ')

    style_prefix = _paragraph_style_to_wt.get(paragraph.style.name)
    if style_prefix:
        wt.append(style_prefix)

    for r in _iterate_over_runs(paragraph, doc_meta):
        if r is None:
            continue

        style_markers = []

        try:
            if r.bold:
                style_markers.append('**')
            if r.italic:
                style_markers.append('//')
            if r.underline:
                style_markers.append('__')
            if r.font.strike:
                style_markers.append('--')
            wiki_color = get_closest_wiki_color(r.font.color)
            if wiki_color and len(r.text) > 0:
                style_markers.append('!!')
        except AttributeError:
            # иногда бывает битый Run у которого нет Pr
            pass

        # иногда бывает Run у которого нет текста

        if r.text is None:
            continue

        # Внутри базовых элементов форматирования по
        # краям не должно быть пробельных символов.
        # Поэтому при наличии пробельных символов по
        # краям текста из run выносим эти символы
        # снаружи форматирования.
        lstripped, rstripped = _stripped(r.text)

        if lstripped:
            wt.append(lstripped)

        cleaned_text = r.text.strip()
        if cleaned_text:
            for marker in style_markers:
                wt.append(marker)
                if marker == '!!':
                    wt.append('(%s)' % wiki_color)
            if hasattr(r, 'link'):
                wt.append('(({link} {text}))'.format(link=r.link, text=r.text))
            else:
                wt.append(cleaned_text)
            for marker in reversed(style_markers):
                wt.append(marker)

        if rstripped:
            wt.append(rstripped)

    wt.append('\n')


def _add_images(doc, page, user, wt):
    images = [s for s in doc.inline_shapes if s.type == WD_INLINE_SHAPE.PICTURE]
    if len(images) > 0:
        wt.append('\n== <<Imported images>>\n')
    for image in images:
        image_part = doc.part.related_parts[image._inline.graphic.graphicData.pic.blipFill.blip.embed]
        image_blob = image_part.blob
        image_io = BytesIO(image_blob)
        image_name = image._inline.graphic.graphicData.pic.nvPicPr.cNvPr.name

        if image_name is None or image_name == '':
            # В атрибуте name не оказалось имени файла,
            # берем имя файла из атрибута descr
            el = etree.fromstring(image._inline.graphic.graphicData.pic.nvPicPr.cNvPr.xml)
            if 'descr' in el.attrib:
                descr = el.attrib['descr']
                image_name = descr.split(':')[-1]
                image_name = image_name.split('/')[-1]

        if image_name is None or image_name == '':
            image_name = 'image-{}'.format(uuid4())

        image_name = translit(image_name)

        if '.' not in image_name:
            image_ext = image_part.image.ext
            image_name = '{}.{}'.format(image_name, image_ext)

        storage_id = files_logic.upload_file_data_to_storage(image_io, len(image_blob), image_name)
        attached_file = files_logic.add_file_to_page(page, user, storage_id)
        wt.append('file:{}\n'.format(attached_file.name))


# В python-docx нет функции, итерирующейся по параграфам
# и таблицам в правильном порядке, есть отдельно
# doc.paragraphs и doc.tables.
# https://github.com/python-openxml/python-docx/pull/395
# TODO: если это реализуют в python-docx, удалить эту функцию
def _iterate_over_paragraphs_and_tables(doc):
    def make_element(el, s):
        if isinstance(el, CT_P):
            return Paragraph(el, doc)
        elif isinstance(el, CT_Tbl):
            return Table(el, s)

    return [make_element(el, doc) for el in doc._body._element if isinstance(el, CT_P) or isinstance(el, CT_Tbl)]


# В python-docx нет функции, итерирующейся по фрагментам текста
# и гиперссылкам в параграфе, фрагменты текста берутся из
# paragraph.runs, адреса гиперссылок достаются через doc.part.rels,
# а через XML параграфа можно получить текст гиперссылки.
# Поэтому для получения адреса гиперссылки используем специальный
# хелпер DocMeta, в который заранее сложены адреса всех найденных
# гиперссылок в документе. В случае гиперссылки навешиваем на
# объект Run свой атрибут link, по которому потом сформируем
# ссылку через ((<ссылка> текст)).
# TODO: если это реализуют в python-docx, удалить эту функцию
def _iterate_over_runs(paragraph, doc_meta):
    hyperlink_tag = '{http://schemas.openxmlformats.org/wordprocessingml/2006/main}hyperlink'

    def make_element(el):
        if isinstance(el, CT_R):
            return Run(el, paragraph)
        elif el.tag == hyperlink_tag:
            link = doc_meta.get_next_hyperlink()
            if len(el.getchildren()) > 0:
                run = Run(el.getchildren()[0], paragraph)
                if link is not None:
                    run.link = link
                return run

    return [make_element(el) for el in paragraph._element if isinstance(el, CT_R) or el.tag == hyperlink_tag]


class _DocMeta:
    def __init__(self, hyperlinks):
        self.hyperlinks = hyperlinks
        self.next_hyperlink_index = 0

    def get_next_hyperlink(self):
        self.next_hyperlink_index -= 1
        try:
            return self.hyperlinks[self.next_hyperlink_index]
        except IndexError:
            return None


# @todo мб регулярочки?


def _lstripped(s):
    return ''.join(takewhile(type(s).isspace, s))


def _rstripped(s):
    return ''.join(reversed(tuple(takewhile(type(s).isspace, reversed(s)))))


def _stripped(s):
    return _lstripped(s), _rstripped(s)
