import waffle

from logging import getLogger

from django.conf import settings
from django.contrib.auth import get_user_model

from wiki.api_core.waffle_switches import ENABLE_COPY_ON_WRITE
from wiki.org import get_org, org_ctx
from wiki.pages.api import save_page
from wiki.pages.logic.clone import clone_page
from wiki.pages.logic.edit import create as create_page
from wiki.pages.logic.etalon import convert_supertag, get_etalon_pages
from wiki.pages.models import Page
from wiki.sync.connect.models import Organization
from wiki.utils.wiki_robot import get_wiki_robot

logger = getLogger(__name__)

CLUSTERS = ('homepage', 'sandbox', 'clusterusers')


class WikiRobotRequiredException(Exception):
    pass


def init_org_data(org_obj):
    """
    Инициализирует данные после импорта организации из Директории.
    """
    create_pages(org_obj)


def create_pages(org_obj):
    org = Organization.objects.get(dir_id=org_obj['id'])

    wiki_robot = get_wiki_robot(organization=org)
    if wiki_robot is None:
        wiki_robot = get_user_model().objects.get(dir_id=settings.ROBOT_DIR_ID)
        wiki_robot.orgs.add(org)

    if waffle.switch_is_active(ENABLE_COPY_ON_WRITE):
        return

    etalon_pages = get_etalon_pages(CLUSTERS, org.lang)
    with org_ctx(org):
        if etalon_pages:
            for page in etalon_pages:
                clone_page(
                    page=page,
                    new_tag=convert_supertag(page.supertag),
                    authors=[wiki_robot],
                    last_author=wiki_robot,
                    is_autogenerated=True,
                )
        else:
            for supertag in CLUSTERS:
                create_page(tag=supertag, user=wiki_robot, body='', title=supertag, notify=False)

        _create_users_cluster(wiki_robot)


def _create_users_cluster(user):
    page = Page(
        tag='users',
        supertag='users',
        owner=user,
        org=get_org(),
        is_autogenerated=True,
    )

    page._should_skip_reserved_protection = True

    page, _, _ = save_page(
        page,
        unicode_text='',
        title='users',
        user=user,
    )

    page.authors.add(user)

    if Page.active.filter(supertag='clusterusers', org=get_org()).exists():
        target = Page.active.get(supertag='clusterusers', org=get_org())
        page.redirects_to = target
        page.save()
