
from datetime import timedelta

from django.conf import settings
from django.utils.translation import ugettext, ugettext_lazy, ungettext

from wiki.notifications.generators.base import EventTypes
from wiki.notifications.models import PageEvent
from wiki.org import get_org
from wiki.pages.cluster import M_CANT_OVERWRITE, M_HOME_PAGE_FORBIDDEN, MESSAGES_MOVE, Cluster
from wiki.pages.models import Page
from wiki.utils import timezone
from wiki.utils.supertag import translit


def log_events(user, move_request, move_result, multiple_pages):
    result_destination = list(move_result.values())
    pages = Page.objects.filter(supertag__in=result_destination, org=get_org())
    inverted_request = dict([(translit(v[1]), v[0]) for v in iter(move_request.items())])
    for destination in result_destination:
        dest_page = None
        for page in pages:
            if page.supertag == destination:
                dest_page = page
                break
        if dest_page:
            event = PageEvent(
                event_type=EventTypes.move_cluster,
                timeout=timezone.now() + timedelta(minutes=20),
                page=dest_page,
                author=user,
                notify=True,
            )

            with_children = multiple_pages.get(inverted_request[destination])
            event.meta = {
                'move_supertag_to_tag': move_request,
                'from': inverted_request[destination],
                'to': dest_page.tag,
                'with_children': with_children,
            }
            event.save()


def move_clusters(user, clusters, with_children=False):
    """
    Move pages (possibly with children) to a new location.
    Returns operation code (similar to http status codes, 200 - success) and
    relevant data.

    @type user: User
    @param user: user that executes move operation
    @type clusters: dict
    @param clusters: словарь "откуда" - "куда". может содержать урлы, а не только супертеги.
    @param with_children: move subpages?
    @rtype: tuple
    @return: (status_code, operation data)
    """

    def cluster_to_multiple_pages(supertags, with_children):
        result = {}
        for supertag in supertags:
            if not with_children:
                result[supertag] = False
            else:
                result[supertag] = Page.objects.filter(supertag__startswith=supertag + '/', org=get_org()).exists()
        return result

    multiple_pages_mapping = cluster_to_multiple_pages(clusters, with_children)

    for key, value in clusters.items():
        if any(tag.lower() == settings.MAIN_PAGE for tag in (key, value)):
            return 409, {
                'message': str(MESSAGES_MOVE.get(M_HOME_PAGE_FORBIDDEN, ugettext_lazy('Unknown error'))),
                'pages': [],
            }

    c = Cluster(user.username)

    answer = c.move(clusters, with_children, allow_reserved_pages=False)
    if answer[0] == 200:
        data = answer[1]
        log_events(user, clusters, data, multiple_pages_mapping)
    else:
        if answer[1] != M_CANT_OVERWRITE:
            message_string = str(MESSAGES_MOVE.get(answer[1], ugettext_lazy('Unknown error')))
        else:
            if not with_children and len(clusters) == 1:
                message_string = ugettext('moveCantOverwriteSinglePage')
            else:
                message_string = ugettext('moveCantOverwriteCluster')
                message_string += ' ' + ungettext(
                    'moveAlreadyHasPageSingular', 'moveAlreadyHasPagesPlural', len(answer[2])
                )
            message_string += ' ' + ugettext('moveChangeAddressToContinue')
        data = {'message': message_string, 'pages': answer[2]}
    return answer[0], data
