from datetime import timedelta

from wiki.notifications.generators.base import EventTypes
from wiki.notifications.models import PageEvent
from wiki.pages import signals
from wiki.pages.models import Page
from wiki.utils import timezone


def change_owner_for_page(page, user, new_owner):
    old_owner = page.owner
    _set_owner_for_pages([page], new_owner)
    create_change_owner_event(page, user, old_owner, new_owner, with_children=False)


def change_owner_for_cluster(root, user, new_owner, old_owner=None):
    old_owner = old_owner or root.owner

    descendants = root.descendants.only('id', 'owner').filter(owner=old_owner)
    old_owner_pages = [root] + list(descendants)

    accessed_subpages = [page for page in old_owner_pages if can_change_owner(page, user)]
    _set_owner_for_pages(accessed_subpages, new_owner)
    create_change_owner_event(root, user, old_owner, new_owner, with_children=len(accessed_subpages) > 1)


def can_change_owner(page, user):
    from wiki.pages import access

    return access.is_admin(user) or user == page.owner


def _set_owner_for_pages(pages, new_owner):
    Page.objects.filter(id__in=[p.id for p in pages]).update(owner=new_owner)
    for page in pages:
        page.authors.clear()
        page.authors.add(new_owner)
    signals.access_changed.send(sender=None, page_list=pages)


def create_change_owner_event(page, user, old_owner, new_owner, with_children, timeout=5):
    PageEvent(
        page=page,
        author=user,
        timeout=timezone.now() + timedelta(minutes=timeout),
        event_type=EventTypes.change_owner,
        notify=True,
        meta={
            'owner_name': new_owner.username,
            'previous_owner': old_owner.username,
            'with_children': with_children,
        },
    ).save()
