import logging

from django.db.models import Q

from wiki.pages.constants import PageOrderPosition
from wiki.sync.connect.base_organization import BaseOrganization

# Не можем расширить диапазон до a-z,
# т.к. не можем полагаться на case sensitive
# сортировки в базе данных
MIN_CHAR = '0'
MAX_CHAR = 'Z'
RANGE_LEN = ord(MAX_CHAR) - ord(MIN_CHAR)

PREFIX_LEN = 20
MIN_PREFIX = '0' * PREFIX_LEN
MAX_PREFIX = '9' * PREFIX_LEN
MAX_STRING_LEN = 255 - PREFIX_LEN

# Пропускаем не буквенно-цифровые символы
SKIPPED_CHARS = {chr(i): 'A' for i in range(58, 65)}

logger = logging.getLogger(__name__)


def middle_string(start, end):
    def get_char(string, index, default):
        return default if index >= len(string) else string[index]

    offset = 0 if start < end else RANGE_LEN

    result = []
    for i in range(max(len(start), len(end)) + 1):
        char_low = get_char(start, i, MIN_CHAR)
        char_high = get_char(end, i, MAX_CHAR)
        if char_low == char_high:
            result.append(char_low)
            continue

        middle_index = int((ord(char_low) + ord(char_high) + offset) / 2)
        if middle_index > ord(MAX_CHAR):
            middle_index -= RANGE_LEN
        middle_char = chr(middle_index)

        if middle_char in SKIPPED_CHARS:
            middle_char = SKIPPED_CHARS[middle_char]
        if middle_char == char_low or middle_char == char_high:
            result.append(char_low)
            continue

        result.append(middle_char)
        break
    result = ''.join(result)

    if len(result) > MAX_STRING_LEN:
        logger.error(f'[RANK] {result} reaches the maximum length, will be truncated')
        result = result[:MAX_STRING_LEN]

    return result


def next_rank(start, end):
    """
    rank представляет собой строку, в состоящую из двух частей, первая часть (префикс),
    равна длине PREFIX_LEN и инкрементируется как целое число от MIN_PREFIX до MAX_PREFIX,
    вторая часть используется в случае, когда нужно получить новый ранг,
    но между префиксами начального и конечного рангов нет свободного пространства
    """
    def _parse_rank(rank, default_prefix, default_char):
        rank_prefix = rank[:PREFIX_LEN]
        if not rank_prefix.isdigit():
            logger.error(
                f'[RANK] {rank[:PREFIX_LEN]} should be digit, '
                f'but it is not (full string {rank}), set to {default_prefix}'
            )
            rank_prefix = default_prefix

        rank_string = rank[PREFIX_LEN:]
        if not rank_string:
            rank_string = default_char

        return rank_prefix, rank_string

    if start == end:
        logger.error(f'[RANK] There is no next string for range from {start} to {end}')
        return start

    if start > end:
        start, end = end, start

    start_prefix, start_string = _parse_rank(start, MIN_PREFIX, MIN_CHAR)
    end_prefix, end_string = _parse_rank(end, MAX_PREFIX, MAX_CHAR)

    format_template = f'{{:0{PREFIX_LEN}}}'
    result = format_template.format(int(start_prefix) + 1)
    if len(result) > PREFIX_LEN or result >= end_prefix:
        result = start_prefix + middle_string(start_string, end_string)

    if not (start <= result <= end):
        logger.error(f"[RANK] This shouldn't be happening, {result} is not between {start} and {end}")

    return result


def next_page_rank(supertag: str, depth: int, organization: BaseOrganization) -> str:
    # Находим значение rank для последней страницы в отсортированном списке
    # в том же кластере, что и страница, для которой расчитываем rank
    path_parts = supertag.split('/')
    page_filter = ~Q(rank=None)
    if len(path_parts) > 1:
        cluster_supertag = '/'.join(path_parts[:-1])
        page_filter &= Q(supertag__startswith=f'{cluster_supertag}/')
    else:
        # Корень
        page_filter &= ~Q(supertag__startswith='/')

    cluster_last_page = (
        organization.get_active_pages()
        .filter(page_filter, depth=depth)
        .order_by('rank').last()
    )
    if not cluster_last_page:
        # первая страница в кластере
        return next_rank(MIN_PREFIX, MAX_PREFIX)

    return next_rank(cluster_last_page.rank, MAX_PREFIX)


def calculate_page_rank(position: PageOrderPosition, supertag: str, organization: BaseOrganization) -> str:
    # Расчитывает новый ранг, чтобы расположить страницу под или над страницей с супертегом supertag
    page = organization.get_active_pages().get(supertag=supertag)
    cluster_supertag = '/'.join(page.supertag.split('/')[:-1])

    if position == PageOrderPosition.AFTER:
        order_by = '-rank'
        filter_for_nearest = Q(rank__lt=page.rank)
    else:
        order_by = 'rank'
        filter_for_nearest = Q(rank__gt=page.rank)

    nearest_page = (
        organization.get_active_pages().filter(
            filter_for_nearest,
            supertag__startswith=f'{cluster_supertag}/',
            depth=page.depth)
        .order_by(order_by)
        .first()
    )
    default_prefix = MIN_PREFIX if position == PageOrderPosition.AFTER else MAX_PREFIX
    nearest_rank = nearest_page.rank if nearest_page else default_prefix
    return next_rank(page.rank, nearest_rank)
