from datetime import timedelta
from typing import List

from django.contrib.auth import get_user_model

from wiki.notifications.generators.base import EventTypes
from wiki.notifications.models import PageEvent
from wiki.pages.models import Page
from wiki.pages.tasks.readonly_mode import UpdateReadonlyModeTask
from wiki.utils import timezone
from wiki.utils.db import on_commit


def change_readonly_mode(page: Page, user: get_user_model(), is_readonly: bool, for_cluster: bool):
    """
    Изменить режим read-only у одной страницы или во всем кластере
    """
    if page.is_readonly != is_readonly:
        page.is_readonly = is_readonly
        page.save()
        create_change_readonly_mode_event(page, user, is_readonly, timeout=5)

    if for_cluster:
        child_pages_ids = Page.active.get_descendants(page.supertag).values_list('id', flat=True)
        if len(child_pages_ids) > 100:
            # если кластер большой, то выполняем обновление страниц в селери таске
            on_commit(
                lambda: UpdateReadonlyModeTask().apply_async(
                    tuple(), dict(root_page_id=page.id, child_pages_ids=tuple(child_pages_ids),
                                  user_id=user.id, is_readonly=is_readonly)
                )
            )
        else:
            update_readonly_mode(child_pages_ids, user.id, is_readonly)


def update_readonly_mode(pages_ids: List[int], user_id: int, is_readonly: bool):
    Page.active.filter(id__in=pages_ids).update(is_readonly=is_readonly)
    PageEvent.objects.bulk_create(
        [PageEvent(
            page_id=page_id,
            author_id=user_id,
            timeout=timezone.now() + timedelta(minutes=5),
            event_type=EventTypes.enable_page_readonly if is_readonly else EventTypes.disable_page_readonly,
            notify=False,
            meta={},
        ) for page_id in pages_ids])


def create_change_readonly_mode_event(page: Page, user: get_user_model(), is_readonly: bool, timeout: int = 5) -> None:
    """
    :param timeout: таймаут в минутах
    """
    PageEvent(
        page=page,
        author=user,
        timeout=timezone.now() + timedelta(minutes=timeout),
        event_type=EventTypes.enable_page_readonly if is_readonly else EventTypes.disable_page_readonly,
        notify=False,
        meta={},
    ).save()
