
from itertools import groupby

from wiki.pages.models import PageLink, Referer


def get_links_from_wiki_pages(to_page):
    """
    Получить список вики-страниц, ссылающихся на данную.

    @param to_page: Страница, на которую ведут ссылки.
    @type to_page: Page

    @rtype: list
    """
    links = (
        PageLink.objects.select_related('from_page')
        .filter(to_page=to_page, from_page__status__gt=0)
        .order_by('from_page__tag')
    )

    return [link.from_page for link in links]


def get_links_from_external_pages(to_page):
    """
    Получить список внешних страниц, ссылающихся на данную.

    Возвращается список tuple'ов (url, num), где
    url - URL внешней страницы (Referrer из запроса).
    num - число переходов с этого URL на страницу.

    @param to_page: Страница, на которую ведут ссылки.
    @type to_page: Page

    @rtype: list
    """
    page_referrers = Referer.objects.filter(page=to_page).order_by('referer').values_list('referer', flat=True)
    page_referrers = groupby(page_referrers)
    page_referrers = ((k, len(list(g))) for k, g in page_referrers)
    page_referrers = sorted(page_referrers, key=lambda p: p[1], reverse=True)

    return page_referrers
