
import json
import logging
import time

import requests
from django.conf import settings
from requests import ReadTimeout

from wiki.utils import timezone
from wiki.utils.log_debounce import LogDebouncer

logger = logging.getLogger(__name__)

stat_error_debounce = LogDebouncer('hatch_debounce', 60 * 60)


def send_stats(request):
    if not settings.SEND_STATS:
        return

    if request.user.is_anonymous:
        return

    data = _prepare_data(request)

    requests_url = 'https://%s/services/collector' % settings.SEND_STATS_HOST

    post_data = {'event': data}

    data = json.dumps(post_data).encode('utf8')

    auth_header = 'Splunk %s' % settings.SEND_STATS_TOKEN

    headers = {'Authorization': auth_header}

    try:
        response = requests.post(requests_url, data=data, headers=headers, verify=False, timeout=10)
        response_json = json.loads(str(response.text))

        if 'text' in response_json:
            if response_json['text'] != 'Success':
                if stat_error_debounce.trigger():
                    logger.warn('[Hatch]: Bad response: %s', response_json)

    except (requests.ConnectionError, ReadTimeout):
        if stat_error_debounce.trigger():
            logger.warn('[Hatch]: Read timeout')

    except Exception:
        logger.exception('[Hatch]: Unexpected error')


def _prepare_data(request):
    user = request.user
    stat_data = request.data

    meta_data = {
        'staffuid': user.staff.uid,
        'user': user.username,
        'backend_timestamp': int(time.time()),  # здесь дублирование 'status', но это поле уже используют
        # существующие клиенты, и им удобнее именно timestamp в unix_time
        'status': str(timezone.now()),
    }

    if 'HTTP_USER_AGENT' in request.META:
        meta_data['http_user_agent'] = request.META['HTTP_USER_AGENT']
    if 'HTTP_REFERER' in request.META:
        meta_data['referer'] = request.META['HTTP_REFERER']
    if hasattr(settings, 'YAUTH_REAL_IP_HEADERS'):
        for user_ip_header_name in settings.YAUTH_REAL_IP_HEADERS:
            if user_ip_header_name in request.META:
                meta_data['user_ip'] = request.META[user_ip_header_name]
                break

    data = {}
    # не перезаписываем поля метаданных, если есть поля с тем же именем в данных статистики
    data.update(stat_data)
    data.update(meta_data)

    return data
