from wiki.pages.access import get_bulk_access_status
from wiki.pages.models import Page


def _limit_qs(qs, limit=None):
    if limit:
        qs = list(qs[: limit + 1])
        limit_exceeded = len(qs) > limit
        del qs[limit:]
    else:
        qs = list(qs)
        limit_exceeded = False

    return qs, limit_exceeded


# TODO: После удаления старого экшена {{tree}} и старого хендлера Tree:
# удалить модуль wiki.pages.access.accessible_subpages.


def get_subpages(
    page_supertag,
    user,
    max_depth=None,
    limit=1000,
    show_owners=False,
    from_yandex_server=False,
    authors=None,
):
    """
    Вернуть подстраницы указанной страницы.

    Возвращает кортеж (subpages, page_pk_to_access, limit_exceeded):

    subpages - список подстраниц,
    page_pk_to_access - словарь {page primary key -> page access}
    limit_exceeded - если True, то число подстраниц оказалось больше, чем указанный лимит.

    @type page_supertag: str

    @type user: User

    @type max_depth: int | None
    @param max_depth: Страницы, глубина которых в кластере больше этого параметра, не будут включены.
    Если параметр равен None, то страницы возвращаются без ограничения по глубине.

    @type limit: int
    @param limit: Лимит размера списка возвращаемых страниц.

    @type show_owners: bool
    @param show_owners: Если True, то будет произведен явный JOIN логинов авторов страниц.

    @type from_yandex_server: bool
    @param from_yandex_server: Запрос от Я.Сервера.

    @param authors: показывать только страницы, где автором является один из пользователей из переданного списка
    @type authors: list

    @rtype: tuple
    """

    subpages = Page.active.get_descendants(page_supertag, include_page=True)

    if show_owners:
        subpages = subpages.prefetch_related('authors')

    if authors:
        subpages = subpages.filter(authors__username__in=authors)

    # Дерево страниц можно построить с параметрами show_grids=<bool> и show_redirects=<bool>.
    # Параметры указывают, показывать ли в дереве гриды и редиректы.
    # Казалось бы, гриды и редиректы можно было бы исключить уже здесь, на уровне запроса в базу данных.
    # Но исключить их не получится - у исключаемого типа страниц (гридов или редиректов) могут быть подстраницы.
    # Поскольку родительские страницы нужно в любом случае показывать, то получится, что в таком случае
    # параметр show_grids=false или show_redirects=false будет проигнорирован, и страница исключенного типа
    # все равно появится в дереве. Чтобы отличить несуществующие страницы от исключенных из дерева
    # гридов и редиректов, нам придется извлекать их из базы данных, чтобы иметь информацию об их типе.

    if max_depth:
        # depth starts with 1;  length(supertag) - length(replace(supertag, '/', ''))  = from zero
        subpages = subpages.filter(depth__lt=page_supertag.count('/') + 1 + max_depth + 1)
        # Old code:
        # subpages = subpages.extra(
        #     where=[
        #         " length(supertag) - length(replace(supertag, '/', '')) < {0} ".format(
        #             page_supertag.count('/') + max_depth + 1
        #         )
        #     ]
        # )

    subpages, limit_exceeded = _limit_qs(subpages, limit)

    page_pk_to_access = {}

    if not from_yandex_server:
        supertags = [p.supertag for p in subpages]
        access = get_bulk_access_status(supertags, user)
        page_pk_to_access = {p.pk: access[p.supertag] for p in subpages}

    return subpages, page_pk_to_access, limit_exceeded


def get_subpages_tags(root_supertag, authors=None, limit=5000):
    """
    Вернуть тэги страниц из кластера.

    Вовращает кортеж (tags, limit_exceeded):

    tags - отсортированный список тэгов страниц,
    limit_exceeded - если True, то число тэгов оказалось больше, чем указанный лимит.

    @param root_supertag: Супертэг корневой страницы кластера.
    @type root_supertag: str

    @param authors: Если задан, то возвращаются только тэги страницы с данными авторами.
    @type authors: list of User

    @type limit: int
    @param limit: Лимит размера списка возвращаемых тэгов.

    @rtype: tuple
    """

    subpages = Page.active.get_descendants(
        root_supertag,
        include_page=False,
    )

    if authors:
        subpages = subpages.filter(authors__in=authors)

    subpages = subpages.order_by('tag').values('tag')

    subpages, limit_exceeded = _limit_qs(subpages, limit)

    tags = [subpage['tag'] for subpage in subpages]

    return tags, limit_exceeded
