from wiki.favorites_v2.tasks import UpdateWatcherAutofolderTask
from wiki.org import get_org_id
from wiki.pages import access as access_logic
from wiki.pages.access import get_cluster_accessible_pages
from wiki.pages.dao.subscription import (
    _filter_by_pages_and_user,
    create_subscriptions,
    filter_pages_watches,
    filter_user_page_watches,
    get_users_subscribed_to_page,
)
from wiki.pages.logic import hierarchy
from wiki.pages.models import Page, PageWatch
from wiki.subscriptions import logic as new_subscriptions
from wiki.sync.connect.base_organization import as_base_organization
from wiki.users.logic.settings import uses_new_subscriptions
from wiki.utils.db import on_commit
from wiki.utils.subscription import generate_event
from wiki.utils.supertag import translit


def is_subscribed_to(user, pages=None, tags=None):
    """
    Вернуть True, если пользователь подписан на заданные страницы.

    Если нужно проверить одну страницу, используйте is_subscribed_to_page

    @type pages: list
    @type tags: list
    @rtype: bool
    """
    if pages:
        page_watches = filter_user_page_watches(user, pages=pages)
        if len(page_watches) != len(pages):
            return False
    if tags:
        page_watches = filter_user_page_watches(user, supertags=list(map(translit, tags)))
        if len(page_watches) != len(tags):
            return False

    return True


def is_subscribed_to_page(user, page_or_tag):
    """
    Вернуть True, если пользователь подписан на страницу.

    @type page_or_tag: Page|basestring
    @rtype: bool
    """
    if isinstance(page_or_tag, Page):
        page = page_or_tag
    elif isinstance(page_or_tag, str):
        try:
            page = Page.objects.get(supertag=translit(page_or_tag))
        except Page.DoesNotExist:
            return False
    else:
        raise TypeError('page_or_tag is typeof {0}'.format(type(page_or_tag)))

    if uses_new_subscriptions(user):
        return new_subscriptions.is_subscribed_to_page(user, page)

    return is_subscribed_to(user, pages=[page])


def get_united_subscribed_users(page):
    subscriptions = new_subscriptions.get_page_subscriptions(page)
    subscribed_users = [s.user for s in subscriptions if uses_new_subscriptions(s.user)]
    watched_users = [user for user in get_users_subscribed_to_page(page) if not uses_new_subscriptions(user)]
    return sorted(subscribed_users + watched_users, key=lambda u: (u.last_name, u.first_name, u.username))


def get_page_watches(page):
    """
    Вернуть список состоящий из подписок на страницу.

    Вспомогательная функция-обертка над filter_page_watches.

    @type page: Page
    @rtype: list
    """
    return filter_pages_watches([page])[page]


def get_user_page_watch(user, page_or_tag):
    """
    Вернуть подписку или None.

    @rtype: PageWatch
    """
    kwargs = {}
    if isinstance(page_or_tag, str):
        kwargs['supertags'] = [translit(page_or_tag)]
    elif isinstance(page_or_tag, Page):
        kwargs['pages'] = [page_or_tag]
    else:
        raise TypeError('page_or_tag is typeof {0}'.format(type(page_or_tag)))
    page_watches = filter_user_page_watches(user, **kwargs)
    if page_watches:
        # всегда один элемент, т.к. подписка на страницу у пользователя всегда только одна
        return page_watches.pop()


def inherit_watches(page, exclude_users=None):
    """
    Наследовать подписчиков от ближайшего родителя для новой страницы.
    """
    organization = as_base_organization(page.org)
    nearest_parent = hierarchy.get_nearest_existing_parent(page.supertag, organization=organization)
    if not nearest_parent:
        return

    watchers = get_users_subscribed_to_page(page=nearest_parent, subscribed_to_cluster=True)
    create_subscriptions(
        [
            PageWatch(
                page=page,
                user=watcher.username,
                is_cluster=True,
            )
            for watcher in watchers
            if not uses_new_subscriptions(watcher) and watcher not in exclude_users
        ],
        ignore_conflicts=True,
    )


def create_watch(page: Page, user, is_cluster: bool):
    create_subscriptions([PageWatch(page=page, user=user.username, is_cluster=is_cluster)], ignore_conflicts=True)


def _produce_subscriptions_for_user_to_pages(pages, user, subscribe_to_cluster=True):
    """
    Вернуть список PageWatch, которые нужно создать чтобы подписать пользователя на pages.

    @rtype: list
    """
    result = []
    pages = set(pages) - {  # убираем все страницы, на которые уже подписаны
        page_watch.page for page_watch in filter_user_page_watches(user, pages)
    }
    access_status = access_logic.get_bulk_access_status([page.supertag for page in pages], user)
    pages = pages - {  # убираем все страницы, к которым нет доступа
        page for page in pages if access_status[page.supertag] < 1
    }
    for page in pages:
        result.append(PageWatch(user=user.username, page=page, is_cluster=subscribe_to_cluster))
    return result


def subscribe_to(pages, *users, **kwargs):
    """
    Подписать пользователей на страницы.

    Если они уже подписаны, повторной подписки создано не будет.
    Если у них нет доступа к странице, подписки создано не будет.

    TODO: текст ниже надо реализовать.
    Для случая, когда подписываем как на кластер
    с флагом subscribe_to_cluster=True, может быть так, что есть
    страницы в pages, на которые уже существует подписка без этого флага.
    Эта функция обновит такие подписки, проставив им этот флаг.

    @return: список созданных объектов PageWatch (подписок).
    @rtype: list
    """
    subscribe_to_cluster = kwargs.get('subscribe_to_cluster', True)
    page_watches = []
    for user in users:  # итерируемся по пользователям, потому что обычно пользователей меньше чем страниц
        # текущие функции в logic не позволяют мне избавиться от этого цикла
        page_watches.extend(
            _produce_subscriptions_for_user_to_pages(pages, user, subscribe_to_cluster=subscribe_to_cluster)
        )
    create_subscriptions(page_watches)
    return page_watches


def subscribe_user_to_cluster(user, cluster_page, author, comment):
    accessible_pages = get_cluster_accessible_pages(cluster_page.supertag, user)

    # Определяем, на какие страницы уже подписан, а на какие надо подписать
    subscriptions = _filter_by_pages_and_user(accessible_pages, user).values('id', 'page__supertag')

    subscribed_to = {sub['page__supertag'] for sub in subscriptions}

    not_subscribed_to = [page for page in accessible_pages if page.supertag not in subscribed_to]

    # Каждую существующую подписку помечаем как "через кластер"
    if len(subscriptions) > 0:
        ids = [str(sub['id']) for sub in subscriptions]
        PageWatch.objects.filter(id__in=ids).update(is_cluster=True)

    # Создаем отсутствующие подписки
    if len(not_subscribed_to) > 0:
        PageWatch.objects.bulk_create(
            [PageWatch(user=user.username, page=page, is_cluster=True) for page in not_subscribed_to],
            ignore_conflicts=True,
        )
        generate_event(
            'subscribe_other_user' if user.id != author.id else 'watch',
            author,
            cluster_page,
            cluster=[page.supertag for page in not_subscribed_to],
            other_subscribed_user=user if user.id != author.id else None,
            comment=comment,
        )

        if not user.is_robot():
            # обновляем закладки в автопапке 'Я наблюдатель'
            update_task = UpdateWatcherAutofolderTask()
            on_commit(lambda: update_task.delay(user_id=user.id, username=user.username, org_id=get_org_id()))

    return len(not_subscribed_to)
