import logging
import re
import unicodedata
from urllib.parse import urlparse

from django.conf import settings
from django.utils.translation import ugettext as _
from rest_framework import serializers

from wiki.utils.errors import ValidationError
from wiki.utils.supertag import remove_trailing_dots, tag_to_supertag

logger = logging.getLogger(__name__)


def compile_regex(regex):
    return re.compile('^%s$' % regex, re.UNICODE)


COMPILED_TAG_REGEX = compile_regex(settings.TAG_REGEX)
COMPILED_TAG_FIRST_SYMBOL_REGEX = compile_regex('|'.join(settings.TAG_VALID_FIRST_SYMBOLS))
COMPILED_TAG_OTHER_SYMBOL_REGEX = compile_regex('|'.join(settings.TAG_VALID_OTHER_SYMBOLS))


# строки, в которых содержится хостнейм фронтэнда
POSSIBLE_HOSTNAMES_STARTERS = list(settings.FRONTEND_HOSTS)
POSSIBLE_HOSTNAMES_STARTERS += ['/' + host for host in POSSIBLE_HOSTNAMES_STARTERS]


def string_to_tag(string):
    """
    Возвращает тэг страницы по произвольной ссылке от пользователя – тэгу или УРЛу.
    Если ссылка не валидна, выбрасывает wiki.utils.errors.ValidationError со значением ссылки в invalid_value.

    Понимает ссылки вида:
        'some/page'
        '/some/page'
        '{wiki_host_name}/some/page'
        '//{wiki_host_name}/some/page'
        '/{wiki_host_name}/some/page'
        'https://{wiki_host_name}/some/page'
    плюс все такие же, только с любым количеством слешей в конце.
    {wiki_host_name} – любой хост из settings.FRONTEND_HOSTS

    @param string: строка, тэг или УРЛ
    @return: супертэг

    @type string: unicode
    @rtype: unicode
    """
    string = _maybe_decode_potential_tag(string)
    original_string = string

    string = _strip_wiki_host(string)

    if not COMPILED_TAG_REGEX.match(string):
        raise ValidationError(original_string)

    return string


def _maybe_decode_potential_tag(string):
    if isinstance(string, bytes):
        msg = 'Non-unicode string passed, it is unsafe. The string: "%s"'
        logger.warning(msg, string)
        # наверняка угадали с кодировкой, но все равно небезопасно.
        string = string.decode('UTF-8')
    return string


def _strip_wiki_host(string, expected_hosts=None):
    """
    Вырезать из строки возможно присутствующие в ней хосты вики.
    """
    wiki_hosts = expected_hosts or settings.FRONTEND_HOSTS
    original_string = string

    string = string.strip()  # удалить пробелы
    parse_result = urlparse(string)  # убрать хостнейм
    if parse_result.hostname:
        if parse_result.hostname not in wiki_hosts:
            if parse_result.scheme:
                raise ValidationError(original_string)
        else:
            string = parse_result.path

    # убрать хостнейм в ссылке вида "{wiki_host_name}/some/page"
    # или "/{wiki_host_name}/some/page"
    for hostname in wiki_hosts:
        possible_beginnings = (
            hostname + '/',
            '/' + hostname + '/',
        )
        if string.startswith(possible_beginnings):
            string = string[len(hostname) + len('/') :]
            break

    # убрать слеши и лишние точки
    string = remove_trailing_dots(string.strip('/'))
    return string


def string_to_supertag(string):
    """
    Возвращает супертэг страницы по произвольной ссылке от пользователя – тэгу или УРЛу.
    """
    return tag_to_supertag(string_to_tag(string))


def clean_tag(string):
    """
    Превратить строку в валидный тег и вернуть. Если не получилось — бросить
    ValidationError.
    """
    try:
        # если сразу более-менее нормальный тег пришел то сразу его вернем
        return string_to_tag(string)
    except ValidationError:
        tag = fix_tag(string)

    if not COMPILED_TAG_REGEX.match(tag):
        logger.warning('Failed to fix tag: {before} -> {after}'.format(before=string, after=tag))
        raise ValidationError(string)
    return tag


def fix_tag(string, expected_hosts=None):
    """
    Пытаемся превратить любую строку в тег.
    """
    string = _maybe_decode_potential_tag(string)
    string = _strip_wiki_host(string, expected_hosts=expected_hosts)
    pieces = string.split('/')
    fixed_pieces = [_fix_tag_piece(piece) for piece in pieces]
    return '/'.join(piece for piece in fixed_pieces if piece)


# некоторые символы будет правильнее заменить, а не вырезать
REPLACEMENTS = {
    ' ': '-',
    unicodedata.lookup('EN DASH'): '-',
    unicodedata.lookup('EM DASH'): '-',
    unicodedata.lookup('NO-BREAK SPACE'): '-',
}


def _fix_tag_piece(string):
    """
    Исправить часть тега (те, которые между слешами —
    может для них есть термин?)
    """
    fixed_tag = ''
    # сначала найдем первый символ, который может быть первым,
    # потом используем вторую регулярку для остальной части
    first_symbol_found = False
    for symbol in string:
        if not first_symbol_found:
            regex = COMPILED_TAG_FIRST_SYMBOL_REGEX
        else:
            regex = COMPILED_TAG_OTHER_SYMBOL_REGEX

        if regex.match(symbol):
            fixed_tag += symbol
            if not first_symbol_found:
                first_symbol_found = True
        elif symbol in REPLACEMENTS:
            fixed_tag += REPLACEMENTS[symbol]
            if not first_symbol_found:
                first_symbol_found = True

    return fixed_tag


def supertag_validator(candidate):
    try:
        string_to_tag(candidate)
    except ValidationError:
        raise serializers.ValidationError(
            # Translators:
            #  ru: В вики не может быть страницы с адресом "%s"
            #  en: There can be no page in Wiki by this address "%s"
            _('There can be no page in Wiki by this address "%s"')
            % candidate
        )
