from multiprocessing import cpu_count

from django.conf import settings
from django.core.management import BaseCommand

from wiki.pages.utils.copy_pages import SkipCheckingAccessToTag, copy_data
from wiki.utils.base_command import argparse_datetime_type

MAX_THREADS = cpu_count()


class Command(BaseCommand):
    help = '''
    Скопировать страницы и ревизии из одного инстанса хранилища в другой.

    Не переливает страницы и их ревизии с ограниченным доступом. Однако можно
    задать исключение вида "homepage/*" - для всех подстраниц homepage, не включая
    саму homepage, или "homepage*" для всех подстраниц, начинающихся с homepage.
    Это не позволяет перелить только homepage и ее подстраницы, но достаточно гибко.

    Пример: $...-manage copy_body_from_storage
            > --from_storage=http://127.0.0.1:89 --from_storage_token=d2lraS13aWtOjQ0Zjk2MTdlMTg1MWI5MGY
            > --revisions=3
            > "homepage*" "wiki/vodstvo*"

    Если не указать подстраницы, которые надо перелить, то по-умолчанию их список
    непуст: он состоит из 'wiki/vodstvo*' и 'homepage*'
    '''
    DEFAULT_ROOTS_TO_COPY = ('wiki/vodstvo*', 'homepage*')

    def add_arguments(self, parser):
        super(Command, self).add_arguments(parser)
        parser.add_argument('--from_storage', dest='from_storage', required=True, help='source storage url')
        parser.add_argument(
            '--to_storage',
            dest='to_storage',
            default=settings.MDS_HOST,
            help='target storage url, defaults to one from wiki instance settings',
        ),
        parser.add_argument(
            '--from_date',
            type=argparse_datetime_type,
            dest='dt_from',
            default=None,
            help='creation/modification datetime of the oldest page to copy, '
            'example: --from_date="2005-12-31 23:59:00"',
        ),
        parser.add_argument(
            '--revisions', type=int, dest='revisions', default=None, help='number of revisions to dump for each page'
        ),
        parser.add_argument('--restricted', action='store_true', dest='restricted'),
        parser.add_argument('--files', action='store_true', dest='files'),
        parser.add_argument(
            '--threads', type=int, default=MAX_THREADS, dest='threads', help='number of threads to spawn'
        )

    def handle(self, *force_these_roots, **options):
        el_from = options.get('from_storage')
        token_from = options.get('from_storage_token')
        el_to = options.get('to_storage')
        token_to = options.get('to_storage_token')
        threads = options.get('threads')
        dt_from = options.get('dt_from')
        revisions_number = options.get('revisions')
        restricted = options.get('restricted')
        files = options.get('files')

        if not force_these_roots:
            force_these_roots = self.DEFAULT_ROOTS_TO_COPY

        # start copying data
        copy_data(
            load_from=el_from,
            token_from=token_from,
            load_to=el_to,
            token_to=token_to,
            dt_from=dt_from,
            revisions_number=revisions_number,
            backup_restricted=restricted,
            backup_files=files,
            threads=threads,
            logger_name='wiki.copy_body_from_storage',
            skip_access_checks_for=[SkipCheckingAccessToTag(root.decode('UTF-8')) for root in force_these_roots],
        )
