
import csv

from django.contrib.auth import get_user_model
from django.core.management.base import BaseCommand, CommandError
from wiki.intranet.models import Staff

from wiki.pages.models import Page
from wiki.utils.supertag import translit


class Error(CommandError):
    pass


class NeedFileNameError(Error):
    pass


class CantOpenFileError(Error):
    pass


class Command(BaseCommand):
    help = 'Find personal clusters and save the list in a CSV file'

    def add_arguments(self, parser):
        super(Command, self).add_arguments(parser)
        parser.add_argument('file', nargs=1, help='CSV filename')

    # TODO(markg): Use only() on models
    def handle(self, *args, **options):
        self.parse_args(options['file'])
        all_users = get_user_model().objects.all()
        for user in all_users:
            self.find_and_save_user_cluster(user)
        if int(options['verbosity']) > 0:
            self.print_statistics()

    def parse_args(self, command_args):
        if len(command_args) < 1:
            raise NeedFileNameError('supply filename')
        file_name = command_args[0]
        try:
            self.file1 = open(file_name, 'w')
        except IOError as io_error:
            message = 'can\'t open file {0!r}: {1}'.format(file_name, io_error)
            raise CantOpenFileError(message)

    def find_and_save_user_cluster(self, user):
        supertag = self.get_supertag(user)
        if supertag is None:
            return
        cluster = self.get_cluster(supertag, user)
        if cluster is None:
            return
        cluster = self.resolve_redirect(cluster, user)
        if cluster is None:
            return
        if cluster.owner != user:
            self.remember_foreign_ownership(user)
            return
        self.get_csv_writer().writerow([user.username, cluster.supertag])
        self.num_ok_users = getattr(self, 'num_ok_users', 0) + 1

    def get_supertag(self, user):
        try:
            staff = user.staff
        except Staff.DoesNotExist:
            self.remember_no_staff(user)
            return None
        return translit(staff.wiki_name)

    def get_cluster(self, supertag, user):
        try:
            cluster = Page.objects.get(supertag=supertag)
        except Page.DoesNotExist:
            pass
        else:
            if cluster.owner == user:
                return cluster
        try:
            cluster = Page.objects.get(supertag=user.username)
        except Page.DoesNotExist:
            pass
        else:
            return cluster
        self.remember_no_cluster(user)

    def resolve_redirect(self, cluster, user):
        try:
            cluster = cluster.redirect_target()
        except Exception:
            self.remember_redirect_error(user)
            return None
        return cluster

    def remember_no_staff(self, user):
        if not hasattr(self, 'no_staff_users'):
            self.no_staff_users = []
        self.no_staff_users.append(user.username)

    def remember_no_cluster(self, user):
        if not hasattr(self, 'no_cluster_users'):
            self.no_cluster_users = []
        self.no_cluster_users.append(user.username)

    def remember_redirect_error(self, user):
        if not hasattr(self, 'redirect_error_users'):
            self.redirect_error_users = []
        self.redirect_error_users.append(user.username)

    def remember_foreign_ownership(self, user):
        if not hasattr(self, 'foreign_ownership_users'):
            self.foreign_ownership_users = []
        self.foreign_ownership_users.append(user.username)

    def get_csv_writer(self):
        if not hasattr(self, 'csv_writer'):
            self.csv_writer = csv.writer(self.file1)
        return self.csv_writer

    def print_statistics(self):
        print('Saved {0} users'.format(getattr(self, 'num_ok_users')))
        no_staff_users = getattr(self, 'no_staff_users', [])
        if no_staff_users:
            print('\nUsers without associated Staff ({0}):'.format(len(no_staff_users)))
            print('  '.join(sorted(no_staff_users)))
            print()
        no_cluster_users = getattr(self, 'no_cluster_users', [])
        if no_cluster_users:
            print(
                ('\nUsers without personal cluster ' 'or already having cluster in /users/* ' '({0}):').format(
                    len(no_cluster_users)
                )
            )
            print('  '.join(sorted(no_cluster_users)))
            print()
        redirect_error_users = getattr(self, 'redirect_error_users', [])
        if redirect_error_users:
            print(('\nUsers whose personal clusters raise ' 'redirect errors ({0}):').format(len(redirect_error_users)))
            print('  '.join(sorted(redirect_error_users)))
            print()
        foreign_ownership_users = getattr(self, 'foreign_ownership_users', [])
        if foreign_ownership_users:
            print(('\nUsers whose clusters belong to somebody ' 'else ({0}):').format(len(foreign_ownership_users)))
            print('  '.join(sorted(foreign_ownership_users)))
            print()
