
import logging

from django.contrib.auth import get_user_model
from django.core.management import BaseCommand
from django.db import transaction
from django.db.models.query_utils import Q

from wiki import access
from wiki.access import NoChanges
from wiki.notifications.models import PageEvent
from wiki.pages.access import get_bulk_raw_access, interpret_raw_access
from wiki.pages.models import Access, Page, Revision
from wiki.utils.models import get_chunked

logger = logging.getLogger('wiki.management_commands')

User = get_user_model()


class Command(BaseCommand):
    """
    Дать доступ стаффовым роботам до вики-страничек старого робота.
    """

    help = 'Create access rights for staff robots'

    def handle(self, *args, **options):
        """
        Может принимать на входе пары имен роботов.
        Разделительимен внутри пары - запятая: старый_робот1,новый_робот1 старый_робот2,новый_робот2
        """
        if args:
            wiki_robot_names = []
            staff_robot_names = []
            for (old_name, new_name) in [item.split(',') for item in args]:
                wiki_robot_names.append(old_name)
                staff_robot_names.append(new_name)
        else:
            wiki_robot_names = [
                '_rpc_noc_robot',
                '_rpc_partner',
                '_rpc_reqwizard',
                '_rpc_spellbuilder',
                '_rpc_statbox',
                '_rpc_maps',
                '_rpc_antispam',
                '_rpc_search_auto_tester',
            ]

            staff_robot_names = [
                'robot-noc',
                'robot-rpc-partner',
                'robot-reqwizard',
                'robot-spellbuilder',
                'robot-rpc-statbox',
                'robot-maps',
                'robot-rpc-antispam',
                'robot-srch-auto-test',
            ]

        wiki_robots = {user.username: user for user in User.objects.filter(username__in=wiki_robot_names)}

        staff_robots = dict([(user.username, user) for user in User.objects.filter(username__in=staff_robot_names)])

        for wiki_robot_name, staff_robot_name in zip(wiki_robot_names, staff_robot_names):
            wiki_robot = wiki_robots[wiki_robot_name]
            staff_robot = staff_robots[staff_robot_name]

            # копируем права на страницы старого робота новому стаффовому роботу.
            print("Copies access rights of '%s' to '%s'" % (wiki_robot_name, staff_robot.username))
            pages_access_list = Access.objects.filter(user_id=wiki_robot.id)
            for pages_access in pages_access_list:
                with transaction.atomic():
                    Access.objects.get_or_create(page_id=pages_access.page_id, staff_id=staff_robot.staff.id)

            # меняем владельца у страниц старого робота на стаффового робота
            print("Change the page owner for pages of '%s' robot" % wiki_robot_name)
            pages_query_set = Page.active.filter(owner_id=wiki_robot.id)

            for subset in get_chunked(pages_query_set):
                raw_access_list = get_bulk_raw_access(subset)
                for page, raw_access in raw_access_list.items():
                    with transaction.atomic():
                        if not Page.objects.filter(id=page.id, owner_id=staff_robot.id).exists():
                            page.owner = staff_robot
                            page.save()

                            # дать доступ старому роботу к его бывшим страницам
                            interpreted_access = interpret_raw_access(raw_access)
                            try:
                                access.set_access(
                                    page,
                                    access.TYPES.RESTRICTED,
                                    get_user_model().objects.get(username='robot-vika'),
                                    staff_models=interpreted_access['users'],
                                    groups=interpreted_access['groups'],
                                )
                            except NoChanges:
                                print('No changes to access: page="%s", robot="%s"' % (page.id, wiki_robot.username))

            print("Change the revision owner for '%s' robot" % wiki_robot_name)
            Revision.objects.filter(author_id=wiki_robot.id).update(author=staff_robot.id)

            print("Change the page_event owner for '%s' robot" % wiki_robot_name)
            # для редактирования и создания меняем автора, остальные удаляем
            edition_types = [PageEvent.EVENT_TYPES.create, PageEvent.EVENT_TYPES.edit]
            PageEvent.objects.filter(event_type__in=edition_types, author_id=wiki_robot.id).update(
                author=staff_robot.id
            )
            PageEvent.objects.filter(~Q(event_type__in=edition_types), author_id=wiki_robot.id).delete()

            print("Change the last_author for '%s' robot" % wiki_robot_name)
            Page.objects.filter(last_author_id=wiki_robot.id).update(last_author=staff_robot.id)
