
import datetime
import logging

from django.contrib.auth import get_user_model
from django.core.management.base import BaseCommand
from django.db.models import F, Q
from wiki.intranet.models import Staff
from wiki.legacy.slave import is_slave

from wiki.utils import timezone

User = get_user_model()

logger = logging.getLogger('wiki.management_commands')


def update_changed_logins():
    """ Всем пользователям, у которых поменялись логины, выставляем обновленный логин """
    qs = Staff.objects.exclude(login=F('user__username'))  # получаем INNER JOIN
    # по пользователям с несовпадающими логинами
    for staff_login, user_login in qs.values_list('login', 'user__username'):
        User.objects.filter(username=user_login).update(username=staff_login)


class Command(BaseCommand):
    help = 'Link model pages.User to wiki.intranet.Staff'
    requires_system_checks = False

    def add_arguments(self, parser):
        super(Command, self).add_arguments(parser)
        parser.add_argument(
            '--force',
            dest='force',
            action='store_true',
            default=False,
            help='Force relink all wiki users to staff data',
        )

    def handle(self, *args, **options):
        try:
            self.do_handle(*args, **options)
        except Exception as e:
            logger.exception('Unhandled error at linkuserstaff command "%s"', repr(e))

    def do_handle(self, *args, **options):
        force = options.get('force')
        verbosity = int(options.get('verbosity'))

        if is_slave():
            if verbosity > 0:
                print('will not execute on slave')
            return

        update_changed_logins()

        if not force:
            # Staff side update
            for staff in Staff.objects.filter(
                Q(modified_at__gte=timezone.now() - datetime.timedelta(0, 3600)) | Q(user=None)
            ):
                if staff.user is None:
                    staff.user, is_created = User.objects.get_or_create(
                        username=staff.login, defaults={'email': staff.get_email()}
                    )
                    staff.save()
                    if is_created:
                        logger.warning('No django.contrib.auth.models.User "%s", had to create it', staff.login)
        else:
            # Wiki side full update
            for user in User.objects.all():
                try:
                    staff = Staff.objects.get(login=user.username)
                    if staff.user.id != user.id:
                        staff.user = user
                        staff.save()
                except Staff.DoesNotExist:
                    if verbosity:
                        print('No wikiname at staff for user "%s"' % user.username)
                except Staff.MultipleObjectsReturned:
                    if verbosity:
                        print(
                            'Two or more users with same wikiname (%s): %s'
                            % (user.username, Staff.objects.filter(wiki_name=user.username))
                        )
