from django.core.management.base import BaseCommand, CommandError

from wiki.pages.models import Page
from wiki.pages.utils.remove import delete_page
from wiki.utils.supertag import translit


def _list_supertags(supertags):
    for supertag in supertags:
        print('  * %s' % supertag)


class Command(BaseCommand):
    help = 'Remove pages listed in stdin'

    def add_arguments(self, parser):
        super(Command, self).add_arguments(parser)
        parser.add_argument('-f', '--file', dest='file', action='store', help='file containing supertags to delete')

    def handle(self, *args, **options):
        if not options.get('file'):
            raise CommandError('You must provide a filename')

        with open(options.get('file')) as f:
            supertags = f.readlines()
        supertags = [translit(s.strip()) for s in supertags if s.strip()]

        if not supertags:
            raise CommandError('Empty file')

        print('The following pages are going to be deleted:')
        _list_supertags(supertags)
        if input('Are you sure? [y/N] ') not in ('y', 'Y'):
            print('Cancelled by user')
            return

        found_pages = []
        not_found_pages = []
        for supertag in supertags:
            try:
                page = Page.objects.get(supertag=supertag)
            except Page.DoesNotExist:
                not_found_pages.append(supertag)
                continue

            delete_page(page)
            found_pages.append(supertag)

        if found_pages:
            print('%d pages have been deleted:' % len(found_pages))
            _list_supertags(found_pages)
        if not_found_pages:
            print('WARNING: %d pages have not been found:' % len(not_found_pages))
            _list_supertags(not_found_pages)
