from datetime import timedelta

from django.contrib.auth import get_user_model
from django.core.management.base import BaseCommand
from wiki.notifications.generators.base import EventTypes
from wiki.notifications.models import PageEvent
from wiki.pages import signals
from wiki.pages.access import has_access
from wiki.pages.models import Page
from wiki.utils import timezone
from wiki.utils.staff import get_staff_person_repo
from wiki.utils.wiki_robot import get_wiki_robot

User = get_user_model()

ROBOT_VIKA = get_wiki_robot()


class Command(BaseCommand):
    help = """
    Команды для сбора инфрормации о страницах бывших сотрудников
    и передачи владения страницами от бывших сотрудников актуальным владельцам.
    """

    def add_arguments(self, parser):
        super(Command, self).add_arguments(parser)
        parser.add_argument('-f', '--func', dest='func', required=False, help='function name for execute')

    def handle(self, *args, **options):
        pages = (
            Page.active.filter(owner__staff__is_dismissed=True)
            .exclude(supertag__startswith='users/')
            .order_by('supertag')
            .values_list('id', 'supertag', 'owner__username')
        )

        print(('Select pages with dismissed owners. Pages count: {}'.format(pages.count())))

        function = options.get('func')
        module = __import__(self.__module__)
        func = getattr(module, function)
        try:
            func()
        except TypeError:
            func(pages)


def process_pages_by_heads(file_name='pages_for_process_by_heads.csv'):
    """
    Прочитать данные из csv файла с форматом данных "cluster;supertag;old_owner;head".
    Выполнить смену владельцев страниц с 'old_owner' на 'head'.
    """
    pages_for_process_by_heads = parse_file(file_name)
    pages_count = 0
    for cluster, supertag, old_owner, head in pages_for_process_by_heads:
        if change_page_owner(supertag, old_owner, head, print_to_log=True):
            pages_count += 1

    print(('\nChanges page owners for {} pages'.format(pages_count)))


def select_pages_for_process_by_heads():
    """
    Прочитать данные из файлов pages_with_new_owners.csv и pages_with_owners_and_heads.csv.
    В первом файле есть перечень владельцев кластеров, содержащих страницы уволенных сотрудников, а во втором есть
    перечень руководителей уволенных сотрудников, чьи страницы не попали в первый файл.
    Сделать выборку страниц из второго файла, где значение атрибута 'head' пересекается со значением атрибута new_owner
    из первого файла и напечатать результат в файл pages_for_process_by_heads.csv
    """
    pages_with_new_owners = parse_file('pages_with_new_owners.csv')
    new_owners = [new_owner for _, _, _, new_owner in pages_with_new_owners]

    pages_with_owners_and_heads = parse_file('pages_with_owners_and_heads.csv')

    with open('pages_for_process_by_heads.csv', 'w') as file:
        file.write('cluster;page;old_owner;head\n')
        for cluster, supertag, old_owner, head, is_head_has_access_to_page in pages_with_owners_and_heads:
            if is_head_has_access_to_page == 'True' and head in new_owners:
                file.write('{};{};{};{}\n'.format(cluster, supertag, old_owner, head))


def parse_file(file_name):
    """
    Прочитать строки из csv файла с символом ';' в качестве разделителя.
    """
    res = list()
    with open(file_name, 'r') as file:
        for line in file:
            res.append(tuple(line.rstrip().split(';')))
    return res


def collect_page_statistics_with_heads(pages):
    """
    Найти руководителей уволенных сотрудников, для страниц которых не получается определить нового актуального
    владельца по родительской странице или кластеру, проверить, есть ли доступ у руководителя к странице
    уволенного сотрудника и вывести результат в файл pages_with_owners_and_heads.csv.
    """
    owners_with_head = collect_page_owners(pages, write_result=False)
    _, pages_without_new_owners = collect_page_statistics(pages, write_result=False)
    with open('pages_with_owners_and_heads.csv', 'w') as file_res:
        file_res.write('cluster;page;old_owner;head;is_head_has_access_to_page\n')
        for cluster, supertag, owner_name in pages_without_new_owners:
            head = owners_with_head[owner_name]
            has_access_to_page = False
            if head:
                try:
                    new_owner = User.objects.get(username=head)
                    has_access_to_page = has_access(supertag, new_owner)
                except User.DoesNotExist:
                    pass
            file_res.write('{};{};{};{};{}\n'.format(cluster, supertag, owner_name, head, has_access_to_page))


def collect_page_owners(pages, write_result=True):
    """
    Найти руководителей уволенных сотрудников, для страниц которых не получается определить нового актуального
    владельца по родительской странице или кластеру и вывести результат в файл old_owners_with_head.csv.
    """
    owners_with_head = {'kolomeetz': 'puroman', 'mixael': 'dmirain', 'petrunina': 'alexkoshelev', 'zhora': 'aliho'}
    for _, _, owner_name in pages:
        if owner_name not in owners_with_head:
            owners_with_head[owner_name] = get_head_of_person(owner_name)

    if write_result:
        write_page_owners_to_file(owners_with_head)

    return owners_with_head


def write_page_owners_to_file(owners_with_head):
    with open('old_owners_with_head.csv', 'w') as file:
        file.write('old_owner;head\n')
        for old_owner in sorted(owners_with_head):
            file.write('{};{}\n'.format(old_owner, owners_with_head[old_owner]))


def change_page_owners(pages):
    """
    Определить для страниц уволенных сотрудников нового актуального владельца по родительской странице или кластеру и
    сменить владельцев страниц, для которых удалось определить нового владельца.
    """
    pages_count = 0
    for page_id, supertag, owner_name in pages:
        cluster, parent_page_owner_name = get_parent_page_owner(supertag, owner_name, print_to_log=False)
        if parent_page_owner_name:
            if change_page_owner(supertag, owner_name, parent_page_owner_name, print_to_log=True):
                pages_count += 1

    print(('\nChanges page owners for {} pages'.format(pages_count)))


def collect_page_statistics(pages, write_result=True):
    """
    Для всех страниц уволенных сотрудников определить нового актуального владельца по родительской странице
    или кластеру.
    Список страниц, для которых удалось определить новых владельцев, вывести в файл pages_with_new_owners.csv,
    а список остальных страниц уволенных сотрудников вывести в файл pages_without_new_owners.csv.
    """
    pages_with_new_owners, pages_without_new_owners = [], []

    for page_id, supertag, owner_name in pages:
        cluster, parent_page_owner_name = get_parent_page_owner(supertag, owner_name, print_to_log=True)
        if parent_page_owner_name:
            pages_with_new_owners.append((cluster, supertag, owner_name, parent_page_owner_name))
        else:
            pages_without_new_owners.append((cluster, supertag, owner_name))

    if write_result:
        with open('pages_with_new_owners.csv', 'w') as f1:
            f1.write('cluster;page;old owner;new owner\n')
            for cluster, supertag, owner_name, parent_page_owner in pages_with_new_owners:
                f1.write('{};{};{};{}\n'.format(cluster, supertag, owner_name, parent_page_owner))

        with open('pages_without_new_owners.csv', 'w') as f2:
            f2.write('cluster;page;old owner\n')
            for cluster, supertag, owner_name in pages_without_new_owners:
                f2.write('{};{};{}\n'.format(cluster, supertag, owner_name))

    return pages_with_new_owners, pages_without_new_owners


def get_parent_page_supertag(supertag):
    return '/'.join(supertag.split('/')[:-1])


def get_parent_page_owner(supertag, owner_name, print_to_log=True):
    """
    Для переданной страницы уволенного сотрудника определить владельца родительской страницы,
    который является действующим сотрудником и кластер этой страницы.
    """
    cluster = supertag
    parent_page_supertag = get_parent_page_supertag(cluster)
    while parent_page_supertag:
        try:
            parent_page_owner = Page.active.get(supertag=parent_page_supertag).owner

            if (
                parent_page_owner.username != owner_name
                and not parent_page_owner.staff.is_dismissed
                and has_access(supertag, parent_page_owner)
            ):
                return cluster, parent_page_owner.username
        except Page.DoesNotExist:
            if print_to_log:
                print(
                    (
                        'Page <{}> does not exist. Try to get a parent page of "{}".'.format(
                            parent_page_supertag, cluster
                        )
                    )
                )

        cluster = parent_page_supertag
        parent_page_supertag = get_parent_page_supertag(cluster)

    return cluster, None


def get_person_with_chief_role(persons):
    for head_person in persons:
        if head_person['role'] == 'chief':
            return head_person['person']['login']
    return None


def get_head_of_person(username):
    """
    Определить руководителя уволенного сотрудника по данным из стафф АПИ.
    """
    employees_repo = get_staff_person_repo()
    person = employees_repo.get_one({'login': username, '_fields': 'login,department_group'})

    head1, head2 = None, None
    try:
        head1 = get_person_with_chief_role(person['department_group']['department']['heads'])
    except Exception as exc:
        print(exc)

    try:
        head2 = get_person_with_chief_role(person['department_group']['parent']['department']['heads'])
    except Exception as exc:
        print(exc)

    return head1 or head2


def change_page_owner(supertag, old_owner_name, new_owner_name, print_to_log=False):
    """
    Сменить владельца страницы с переданным супертагом.
    """
    try:
        page = Page.active.get(supertag=supertag)
        if page.owner.username != old_owner_name:
            print(
                (
                    'Page "{}" has different owner: "{}", expected "{}"'.format(
                        supertag, page.owner.username, old_owner_name
                    )
                )
            )
            return False
        if not page.owner.staff.is_dismissed:
            print(('Page "{}" has active owner: "{}"'.format(supertag, page.owner.username)))
            return False
    except Page.DoesNotExist:
        print(('Page <{}> does not exist'.format(supertag)))
        return False

    try:
        new_owner = User.objects.get(username=new_owner_name)
    except User.DoesNotExist:
        return False

    if new_owner.staff.is_dismissed:
        print(('User "{}" is already dismissed'.format(new_owner_name)))
        return False

    if not has_access(supertag, new_owner):
        print(('User "{}" has not access to page "{}"'.format(new_owner_name, supertag)))
        return False

    _set_owner_for_pages([page], new_owner)
    create_change_owner_event(page, ROBOT_VIKA, old_owner_name, new_owner, with_children=False)
    if print_to_log:
        print(('Changed owner for page "{}": from "{}" to "{}"'.format(supertag, old_owner_name, new_owner_name)))
    return True


def _set_owner_for_pages(pages, new_owner):
    Page.objects.filter(id__in=[p.id for p in pages]).update(owner=new_owner)
    signals.access_changed.send(sender=None, page_list=pages)


def create_change_owner_event(page, user, old_owner_name, new_owner, with_children, timeout=5):
    """
    Создать событие о смене владельца страницы без отсылки уведомлений пользователям.
    """
    PageEvent(
        page=page,
        author=user,
        timeout=timezone.now() + timedelta(minutes=timeout),
        event_type=EventTypes.change_owner,
        notify=False,
        meta={
            'owner_name': new_owner.username,
            'previous_owner': old_owner_name,
            'with_children': with_children,
        },
    ).save()
