
from django.core.management.base import BaseCommand

from wiki.pages.logic import tags
from wiki.pages.models import Page, PageRename
from wiki.utils.models import queryset_iterator
from wiki.utils.supertag import tag_to_supertag


class Command(BaseCommand):
    """
    Собрать все страницы с плохим тегом и положить их в таблицу PageRename,
    чтобы потом использовать при обходе всех страниц.
    """

    def add_arguments(self, parser):
        super(Command, self).add_arguments(parser)
        parser.add_argument(
            '--reset',
            action='store_true',
            dest='reset',
            default=False,
            help='reset pages_pagerename table before collecting',
        )

    def handle(self, *args, **options):
        if options['reset']:
            PageRename.objects.all().delete()

        pages = queryset_iterator(Page.active.all())
        collected_count = 0
        for page in pages:
            if not is_valid_tag(page):
                collected_count += 1
                mark_for_renaming(page)

        print('Collected %s pages' % collected_count)


def is_valid_tag(page):
    return bool(tags.COMPILED_TAG_REGEX.match(page.tag))


POSSIBLE_WIKI_HOSTS = (
    'wiki.yandex-team.ru',
    'old.wiki.yandex-team.ru',
    'beta.wiki.yandex-team.ru',
    'wiki.test.yandex-team.ru',
    'old.wiki.test.yandex-team.ru',
    'beta.wiki.test.yandex-team.ru',
    'wiki.test.cs.yandex-team.ru',
)


def mark_for_renaming(page):
    old_tag = page.tag
    old_supertag = page.supertag
    try:
        fixed_tag = tags.fix_tag(old_tag, expected_hosts=POSSIBLE_WIKI_HOSTS)
    except Exception as exc:
        print(('Failed to fix tags ' + str(exc)))
        return

    if is_tag_taken(tag=fixed_tag, exclude_page=page):
        print(('Tag %s taken' % fixed_tag))
        for suffix in range(100):
            fixed_suffixed_tag = fixed_tag + '-' + str(suffix)
            if not is_tag_taken(tag=fixed_suffixed_tag, exclude_page=page):
                fixed_tag = fixed_suffixed_tag
                print(('Using tag %s' % fixed_tag))
                break

    PageRename.objects.create(
        page=page,
        old_tag=old_tag,
        old_supertag=old_supertag,
        new_tag=fixed_tag,
        new_supertag=tag_to_supertag(fixed_tag),
    )


def is_tag_taken(tag, exclude_page=None):
    supertag = tag_to_supertag(tag)
    queryset = Page.active.filter(supertag=supertag)
    if exclude_page:
        queryset = queryset.exclude(id=exclude_page.id)
    return queryset.exists()
