import time

from django.core.management.base import BaseCommand
from django.db import transaction, connection

import logging

logger = logging.getLogger('wiki.management_commands')


QUERY = """
WITH page_info AS
(
    SELECT
        page.id AS page_id,
        author_query.id AS author_id
    FROM
        pages_page AS page,
        LATERAL (
            SELECT
                auth_user.id
            FROM
                auth_user
            INNER JOIN
                pages_page_authors AS authors ON (authors.user_id = auth_user.id)
            INNER JOIN
                intranet_staff AS staff ON (auth_user.id = staff.user_id)
            WHERE
                authors.page_id = page.id
            ORDER BY
                staff.is_dismissed, authors.id
            LIMIT 1
        ) AS author_query
    WHERE
        page.id >= %(start_id)s
        AND page.id < %(end_id)s
)

UPDATE
    pages_page
SET
    owner_id = page_info.author_id
FROM
    page_info
WHERE
    pages_page.id = page_info.page_id AND
    (pages_page.owner_id IS NULL OR pages_page.owner_id != page_info.author_id)
"""


class Command(BaseCommand):
    help = 'Set owner as first author'

    def add_arguments(self, parser):
        super(Command, self).add_arguments(parser)
        parser.add_argument('--start-page-id', type=int, default=0)
        parser.add_argument('--end-page-id', type=int, default=5_000_000)
        parser.add_argument('--step', type=int, help='count per one iteration', default=200)
        parser.add_argument('--sleep', type=int, help='time to sleep', default=2)

    def handle(self, *args, **options):
        start_page_id = options.get('start_page_id', 0)
        end_page_id = options.get('end_page_id', 5_000_000) + 1
        step = options.get('step', 200)
        sleep = options.get('sleep', 2)

        logger.info(f'Owner. Set as first author. START pages: [{start_page_id} : {end_page_id}]')

        for i in range(start_page_id, end_page_id, step):
            with transaction.atomic(), connection.cursor() as cursor:
                cursor.execute(QUERY, {'start_id': i, 'end_id': i + step})

            logger.info(f'Owner. Set as first author. pages: [{i} : {i + step}]')
            time.sleep(sleep)

        logger.info(f'Owner. Set as first author. END pages: [{start_page_id} : {end_page_id}]')
