import datetime
import logging

from django.db.models import Count

from wiki.files.models import File
from wiki.notifications.models import PageEvent
from wiki.pages.models import Page, Revision
from wiki.utils.backports.mds_compat import APIError

logger = logging.getLogger(__name__)


def mds_content_used_only_once(mds_storage_id):
    count = Revision.objects.filter(mds_storage_id=mds_storage_id).count()
    return count <= 1


def remove_unused_files(last_modified_date=None):
    """
    Удаляет файлы, помеченных как удаленные, из базы и их контент из хранилища.

    @param last_modified_date: дата последнего изменения файла, которая используется при выборке удаленных объектов
    в качестве фильтра - дата последнего изменения файла должна быть старше переданного в качестве параметра значения.
    По умолчанию конечная дата рассматриваемых изменений - 1 января 2019 года.
    """
    if last_modified_date is None:
        last_modified_date = datetime.date(2019, 1, 1)
    files_ids = File.objects.filter(status=0, modified_at__lte=last_modified_date).values_list('id', flat=True)
    for file_id in files_ids:
        try:
            file = File.objects.get(id=file_id)
            if file.is_active:
                continue
            logger.info('Deleting a file with id=%s and name=%s and url=%s' % (file.id, file.name, file.url))
            file.mds_storage_id.delete(file.mds_storage_id.name)
            super(File, file).delete()
        except Exception:
            logger.exception('Unused files removal failed')


def remove_unused_events_and_revisions(last_modified_date=None):
    """
    Удаляет из базы все события и ревизии (кроме последней) страниц, помеченных как удаленные, а также удаляет
    из хранилища контент этих ревизий. Последняя ревизия не удаляется, чтобы по ней была возможность, если
    понадобиться, восстановить удаленную страницу.

    @param last_modified_date: дата последнего изменения страницы, которая используется при выборке удаленных объектов
    в качестве фильтра - дата последнего изменения страницы должна быть старше переданного
    в качестве параметра значения.
    По умолчанию конечная дата рассматриваемых изменений - 1 января 2019 года.
    """
    if last_modified_date is None:
        last_modified_date = datetime.date(2019, 1, 1)

    pages_ids = Page.objects.filter(status=0, modified_at__lte=last_modified_date).values_list('id', flat=True)

    for page_id in pages_ids:
        logger.info('Deleting all events of page with id=%s' % page_id)
        events_ids = PageEvent.objects.filter(page_id=page_id).values_list('id', flat=True)
        try:
            for event_id in events_ids:
                event = PageEvent.objects.get(id=event_id)
                event.delete()
        except Exception as err:
            logger.exception(err)

        logger.info('Deleting all revisions of page with id=%s' % page_id)
        rev_ids = Revision.objects.filter(page_id=page_id).values_list('id', flat=True).order_by('-id')[1:]
        for rev_id in rev_ids:
            try:
                revision = Revision.objects.get(id=rev_id)
                logger.info('Deleting a revision with id=%s and url=%s' % (revision.id, revision.url))
                if mds_content_used_only_once(revision.mds_storage_id.name):
                    revision.mds_storage_id.delete(revision.mds_storage_id.name)
                else:
                    logger.info("%s used by other revisions, won't delete it" % revision.mds_storage_id.name)
                revision.delete()
            except Exception:
                logger.exception('Unused events cleanup failed')


def remove_old_revisions(page_id, recent_revisions_count=1000):
    """
    Удаляет ревизии у страницы с переданным page_id, если их количество более значения recent_revisions_count.
    У страницы после удаления остаются самые свежие ревизии в количестве, не превышающем значение
    recent_revisions_count.
    """
    rev_ids = (
        Revision.objects.filter(page_id=page_id).values_list('id', flat=True).order_by('-id')[recent_revisions_count:]
    )
    success_count = 0
    fail_count = 0
    last_error = None
    for rev_id in rev_ids:
        try:
            revision = Revision.objects.get(id=rev_id)
            logger.info('Try to delete a revision with id=%s and url=%s' % (revision.id, revision.url))
            if mds_content_used_only_once(revision.mds_storage_id.name):
                try:
                    revision.mds_storage_id.delete(revision.mds_storage_id.name)
                except APIError as api_error:
                    if str(api_error).count('404 Client Error: Not Found') == 0:
                        # если это не ошибка 404, то пропускаем эту ревизию, иначе удаляем ее, так как она оказалась
                        # по какой-то причине без контента
                        fail_count += 1
                        last_error = api_error
                        continue
            else:
                logger.info("%s used by other revisions, won't delete it" % revision.mds_storage_id.name)
            revision.delete()
            success_count += 1
        except Exception as e:
            logger.exception('Old revision removal failed')
            fail_count += 1
            last_error = e

    return success_count, fail_count, last_error


def remove_revisions_of_deleted_pages(recent_revisions_count=50, revisions_count_for_select=100):
    """
    Вычисляет удаленные страницы с максимальным количеством ревизий и удаляет более старые, оставляя более свежие
    в количестве равным значению параметра recent_revisions_count
    """
    revisions = (
        Revision.objects.filter(page__status=0)
        .values('page_id')
        .annotate(rev_num=Count('page_id'))
        .filter(rev_num__gt=recent_revisions_count)[:revisions_count_for_select]
    )

    clear_run = True

    for revision in revisions:
        # у удаленной страницы больше recent_revisions_count ревизий, оставляем только
        # recent_revisions_count самых свежих ревизий, а остальные удаляем

        success, fails, last_error = remove_old_revisions(
            revision['page_id'], recent_revisions_count=recent_revisions_count
        )

        logger.info(
            'Page id={} has {} revisions. Removal excess revisions: success {}, failed {}, last_error {}.'.format(
                revision['page_id'], revision['rev_num'], success, fails, last_error
            )
        )
        clear_run = clear_run and (fails == 0)

    if not clear_run:
        # Мдс никогда не гарантирует того что твой запрос на удаление отработает
        # Если -1 ДЦ, то вообще будут 500, и это "нормально". Чтобы не спамить логи, выводим одну ошибку
        # Кому надо - сходит за деталями

        logger.warning('Some of revisions failed to cleanup. See log for details')
