import warnings

from django.conf import settings
from django.core.cache import cache as project_cache
from django.db import models
from wiki.intranet.models import Staff

from wiki.org import get_org, get_org_id
from wiki.pages.models.page import Page
from wiki.users.models import Group


class AccessManager(models.Manager):
    def supertags(self):
        return (
            self.select_related('page')
            .filter(page__status__gt=0, page__org=get_org())
            .values_list('page__supertag', flat=True)
            .order_by('page__supertag')
        )

    def allowed_logins(self, supertag, cache=project_cache):
        logins = set()
        for access_object in self.filter(page__supertag=supertag, page__org=get_org()):
            allowed_logins = access_object.get_allowed_logins(cache)
            if allowed_logins:
                logins |= set(allowed_logins)
        return logins


class Access(models.Model):
    objects = AccessManager()

    page = models.ForeignKey(Page, on_delete=models.CASCADE)
    staff = models.ForeignKey(Staff, null=True, on_delete=models.CASCADE)
    group = models.ForeignKey(Group, null=True, on_delete=models.CASCADE)
    is_common = models.BooleanField(default=False)
    is_owner = models.BooleanField(default=False)
    is_anonymous = models.BooleanField(default=False)

    def __str__(self):
        return 'Access for page %s' % self.page

    def get_allowed_logins(self, cache=project_cache):
        """
        Вернуть список имен пользователей, которым доступ разрешен или
        None, если доступ не ограничен.

        Эта ручка не используется нигде!
        Планировалось, что она будет нужна в /_api/locked_pages/.
        """
        warnings.warn('Please do not use me', DeprecationWarning)

        if self.is_anonymous or self.is_common:
            return None
        if self.is_owner:
            return [author.username for author in self.page.get_authors()]
        if self.staff:
            return [self.staff.login]
        if self.group:
            key = 'group:{0}:{1}:allowed_logins'.format(self.group.id, get_org_id() or 0)
            allowed_logins = cache.get(key)
            if allowed_logins is None:
                allowed_logins = [member.username for member in self.group.get_all_members()]
                cache.set(key, allowed_logins, 15 * 60)
            return allowed_logins

    class Meta:
        app_label = 'pages'
        # @TODO: valid unique fieldset
        # unique_together = (('page', 'staff', 'group'),)


class AccessRequest(models.Model):
    id: int
    objects: models.Manager

    applicant = models.ForeignKey(settings.AUTH_USER_MODEL, related_name='applicant', on_delete=models.CASCADE)
    page = models.ForeignKey(Page, on_delete=models.CASCADE)
    created_at = models.DateTimeField(auto_now_add=True)
    reason = models.TextField()
    verdict = models.NullBooleanField(null=True)
    verdict_reason = models.TextField(null=True)
    verdict_by = models.ForeignKey(
        settings.AUTH_USER_MODEL, related_name='verdict_by', null=True, on_delete=models.CASCADE
    )

    def __str__(self):
        return '"%s" to "%s"' % (self.applicant.username, self.page.tag)

    class Meta:
        app_label = 'pages'
