from enum import Enum
from typing import Generator

from pydantic import BaseModel, Field

from django.conf import settings
from django.contrib.postgres.fields import JSONField
from django.db import models
from django.db.models import Q
from enumfields import EnumIntegerField

from wiki.api_v2.public.pages.page_identity import PageIdentity
from wiki.api_v2.schemas import Slug
from wiki.async_operations.operation_executors.move_cluster.consts import MoveCluster
from wiki.sync.connect.models import Organization


class ChangeStatus(int, Enum):
    IDLE = 0
    FWD_MOVE_PHASE = 1
    FWD_REWRITE_PHASE = 2
    UNDO_MOVE_PHASE = 3
    UNDO_REWRITE_PHASE = 4
    FWD_CLONE_PHASE = 5


def get_parents(src: str) -> Generator[str, None, None]:
    for i, ch in enumerate(src):
        if ch == '/':
            yield src[:i]
    yield src


class AffectedPage(PageIdentity):
    previous_slug: Slug


class AffectedPages(BaseModel):
    pages: list[AffectedPage] = Field(default_factory=list)


class Operations(BaseModel):
    input: list[MoveCluster]
    compress: list[MoveCluster]


class ClusterChange(models.Model):
    id: models.IntegerField
    objects: models.Manager

    user = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.CASCADE)
    org = models.ForeignKey(Organization, blank=True, null=True, default=None, on_delete=models.CASCADE)
    finished_at = models.DateTimeField(auto_now_add=True)

    affected_pages = JSONField(default=dict)
    operations = JSONField(default=dict)

    def __str__(self):
        org_id = self.org and self.org.id  # noqa
        return f'ClusterChange {self.id}: {self.user.username} at {self.finished_at} in org [{org_id}]'  # noqa


class ClusterBlock(models.Model):
    id: models.IntegerField
    objects: models.Manager

    task_id = models.CharField(max_length=255, db_index=True)
    user = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.CASCADE)
    org = models.ForeignKey(Organization, blank=True, null=True, default=None, on_delete=models.CASCADE)

    status = EnumIntegerField(ChangeStatus)
    source = models.CharField(max_length=255, db_index=True)
    target = models.CharField(max_length=255, db_index=True)

    @classmethod
    def find_blocked(cls, slugs: list[str], org_id: int | None, in_children: bool = True) -> list['ClusterBlock']:
        parents = set()
        for slug in slugs:
            parents.update(get_parents(slug))
        query = Q(source__in=parents) | Q(target__in=parents)

        if in_children:
            for slug in slugs:
                query |= Q(source__startswith=slug + '/') | Q(target__startswith=slug + '/')

        return list(cls.objects.filter(query, org_id=org_id))

    def __str__(self):
        org_id = self.org and self.org.id  # noqa
        return f'ClusterBlock {self.id}: {self.status.name}: "{self.source}" -> "{self.target}" in org [{org_id}]'
