import logging

from django.conf import settings
from django.db import models
from django.db.models.query import QuerySet
from django.db.models.signals import post_save
from django.dispatch import receiver

from wiki.pages.models.page import Page

logger = logging.getLogger('wiki.models.comment')

reverse_xrange = lambda n: range(n - 1, -1, -1)  # n-1, n-2, .. 0


class CommentQuerySet(QuerySet):
    def for_page(self, page):
        return self.filter(page=page)

    def active(self):
        return self.filter(status=True)

    def with_related(self):
        return self.select_related('parent', 'page', 'user')


class Comment(models.Model):
    page = models.ForeignKey(Page, on_delete=models.CASCADE)
    user = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.CASCADE)
    parent = models.ForeignKey('self', null=True, on_delete=models.CASCADE)
    body = models.TextField()
    created_at = models.DateTimeField(auto_now_add=True)
    # дата модификации страницы на момент создания комментария
    page_at = models.DateTimeField()
    status = models.BooleanField(default=True)  # True - активен, False - удален

    objects = CommentQuerySet.as_manager()

    def mark_removed(self):
        """
        Пометить комментарий удаленным
        """
        self.status = False

    def mark_active(self):
        """
        Пометить комментарий как активный (не-удаленный)
        """
        self.status = True

    @classmethod
    def get_page_comments(cls, page):
        """
        Вернуть все комментарии для страницы
        @param page: страница, для которой запрошены комментарии
        @rtype: QuerySet
        """
        # запрашиваем все комментарии страницы, отсортированные по родителю и в порядке возрастания
        # даты создания

        comments = list(cls.objects.filter(page=page).select_related('parent', 'page', 'user'))
        comments.sort(key=lambda c: c.id)
        comments.sort(key=lambda c: c.parent.id if c.parent else 0)

        return comments

    @staticmethod
    def make_list_for_template(comments):
        """
        Построить список комментариев для шаблона.
        Комментарии могуть быть вложенными. Дети идут в списке следом за родителями.
        Удаленные комментарии показываются, если у них есть не удаленные дети.
        Порядок объектов в возвращаемом списке подробно описан в тесте
        pages.comment.CommentsTest.test_comment_structure

        @type comments: list
        @param comments: список комментариев, отсортированный по родителю и ASC дате создания
        @rtype: list
        """
        level = 1  # уровень вложенности комментариев, начнем сразу с 1го
        result = []  # итоговый список комментариев, дети идут за родителями

        unprocessed = []  # начальный список необработанных комментариев
        to_remove = set()  # здесь будут все кандидаты на удаление из итогового дерева
        level_parents = []  # родители для очередного уровня вложенности дерева комментариев,
        # изначально -- комментарии 0го уровня.

        # заполним исходные данные
        for comment in comments:
            # вспомогательные данные для построения отступов на фронтэнде
            comment.offset_level = 0

            # занесем всех co статусом False в кандидаты на удаление из итогового дерева
            if not comment.status:
                to_remove.add(comment)

            # сразу заполняем 0й уровень комментариями без родителей
            if not comment.parent:
                level_parents.append(comment)
                result.append(comment)
            else:
                # остальных -- в очередь на обработку
                unprocessed.append(comment)

        level_parents = set(level_parents)

        # Нам дано дерево комментариев в виде списка элементов с указанными родителями.
        # Прокручиваемся по списку, через родителей находя комментарии с одного уровня дерева, и раскладывая
        # в новый список следом за их родителями
        while True:
            level_comments = []  # здесь собираются все комментарии текущего уровня дерева
            next_comments = []  # здесь собираются не обработанные на итерации комментарии

            # попробуюем найти в очереди на обработку комментарии следующего уровня
            for comment in unprocessed:
                if comment.parent in level_parents:
                    # переносим отобранный комментарий в список родителей на следующую итерацию
                    level_comments.append(comment)

                    # если у комментария со статусом True есть родитель из кандидатов
                    # на удаление -- родителя удалять не будем.
                    if comment.status and comment.parent in to_remove:
                        to_remove.remove(comment.parent)

                    # вспомогательные данные для построения отступов на фронтэнде
                    comment.offset_level = level * 20

                else:
                    # комментарии с неподходящими родителями уходят как необработанные на следующий круг
                    next_comments.append(comment)

            # если в этой итерации никого не отобрано -- комментарии в дереве должны были кончится
            if not level_comments:
                # если остались комментарии вне дерева -- значит мы изначально выбрали лишнего
                if unprocessed:
                    raise RuntimeError(
                        'Couldn\'t make comments with ids={0} into tree'.format([c.id for c in unprocessed])
                    )

                # иначе -- выходим из цикла нормально
                break

            # засунем отобранные комментарии в результат на верные места
            for index in reverse_xrange(len(level_comments)):
                comment = level_comments[index]
                # комментарии должны идти по порядку после родителей
                parent_index = result.index(comment.parent)  # индекс родителя в результате
                result.insert(parent_index + 1, comment)  # вставляем комментарий следующим за родителем

            # заход на следующую итерацию
            unprocessed = next_comments
            level_parents = set(level_comments)  # по set() искать быстрее
            level += 1

        # удаляем из результата всех дошедших до этого этапа кандидатов
        for comment in to_remove:
            result.remove(comment)

        return result

    @staticmethod
    def populate_comments_deletion_flag(comments, user, is_admin=False):
        """
        Выставить у комментариев из переданного списка признак возможности удаления
        указанным пользователем

        @type comments: list
        @param comments: список объектов комментариев
        @type user: django.contrib.auth.models.User
        @param user: пользователь
        @param is_admin: показатель того, что пользователь -- админ
        """

        # для комментариев выставляется флаг, можно ли их удалять указанному пользователю
        for comment in comments:
            comment.may_be_deleted = is_admin or comment.user.username == user.username

    class Meta:
        db_table = 'wakka_comments'
        app_label = 'pages'

    def __str__(self):
        return "%s: from %s to \"%s\"" % (self.pk, self.user.username, self.page.supertag)


@receiver(post_save, sender=Comment)
def update_page_comments_count(sender, instance, created, **kwargs):
    """
    Обновить счетчик комментариев у страницы
    """
    page = instance.page
    if created:
        page.comments += 1

    else:
        page.comments = Comment.objects.filter(page=page, status=True).count()

    page.save()

    logger.debug('page id={0} comments count was updated to {1}'.format(page.id, page.comments))
