from django.conf import settings
from random import randint

from django.core.files.uploadedfile import SimpleUploadedFile
from django.db import models
from django.utils.timezone import now

from wiki.utils.s3.storage import S3Storage

try:
    from ujson import dumps, loads
except ImportError:
    from json import loads, dumps


class SerializedPropsMixin(models.Model):
    """
    It is mixin for Django models, allowing to operate transparently with some storage.
    For correctly usage derive your model class from this class as in the following example:
    class Page(SerializedPropsMixin, models.Model)
    """

    # List of properties
    _serialized_props = tuple()

    # Dict with defaults for every property or scalar for all
    _serialized_props_defaults = None

    # Have serialized properties been modified?
    _serialized_props_modified = False

    # имя FileField атрибута в модели, к которой подмешан миксин
    _storage_field_name = 'storage_id'

    @property
    def _serialized_mixin_storage_field(self):
        return getattr(self, self._storage_field_name)

    @_serialized_mixin_storage_field.setter
    def _serialized_mixin_storage_field(self, value):
        setattr(self, self._storage_field_name, value)

    @staticmethod
    def _storage_loads(data):
        return loads(data)

    @staticmethod
    def _storage_dumps(data):
        return SimpleUploadedFile('fake_uploaded_file', dumps(data).encode())

    def __getattr__(self, name):
        """
        Returns value of model's field from storage system.
        If field isn't stored into storage, returns default value
        based on model's _serialized_props_defaults property.
        """
        if name in self._serialized_props:
            self._init_data()
            return self._data.get(
                name,
                self._serialized_props_defaults.get(name, None)
                if isinstance(self._serialized_props_defaults, dict)
                else self._serialized_props_defaults,
            )
        return super(SerializedPropsMixin, self).__getattribute__(name)

    def __setattr__(self, name, value):
        """
        Set value for model's field from storage system.
        """
        if name in self._serialized_props:
            self._init_data()
            self._data[name] = value
            self._serialized_props_modified = True
        return super(SerializedPropsMixin, self).__setattr__(name, value)

    def _init_data(self):
        """
        Implements lazy loading of data from storage system.
        """
        if not hasattr(self, '_data'):
            if self._serialized_mixin_storage_field:
                self._data = self._storage_loads(self._serialized_mixin_storage_field.read())
            else:
                self._data = {}

    def _make_storage_id(self):
        """
        Схема storage_id для файлов.

        base64 не использует в качестве буквы своего алфавита двоеточие ":", поэтому
        можно рассчитывать, что число разделителей у результата этой функции
        всегда фиксированное (http://tools.ietf.org/html/rfc3548.html)
        """
        model_name = self._meta.model_name
        datetime_now = now().strftime('%Y-%m-%d %H:%M:%S:%f')
        random_int = str(randint(10 ** 6, 10 ** 7 - 1))

        storage = self._serialized_mixin_storage_field.storage

        if isinstance(storage, S3Storage):
            return f'{settings.WIKI_CODE}/{model_name}/{self.pk}/{datetime_now}:{random_int}'

        # mds storage
        return f'{model_name}:{self.pk}:{datetime_now}:{random_int}'

    def is_props_changed(self):
        """
        Serialized properties was loaded and changed or not.
        """
        return hasattr(self, '_data') and self._serialized_props_modified

    def save(self, *args, **kwargs):
        """
        Saves model's data to storage.
        See also receivers of pre_save and post_save signals in this file.
        """

        # Save serialized properties to storage only if they have been
        # loaded and changed.
        if self.is_props_changed():
            self._serialized_mixin_storage_field = self._serialized_mixin_storage_field.storage.save(
                self._make_storage_id(), self._storage_dumps(self._data)
            )

        res = super(SerializedPropsMixin, self).save(*args, **kwargs)
        if self._serialized_props_modified:
            self._serialized_props_modified = False
        return res

    class Meta:
        abstract = True
