import logging
import waffle
from typing import Union, Dict, List, Any

import six
from django.conf import settings
from django.db import models
from django.db.models import Q
from django.db.models.aggregates import Max
from django.utils.translation import ugettext_lazy

from wiki.api_core.waffle_switches import S3_DEFAULT_STORAGE
from wiki.api_v2.public.pages.schemas import BreadcrumbSchema
from wiki.cloudsearch.indexable_model_mixin import IndexableModelMixin
from wiki.org import get_org, org_ctx
from wiki.pages.constants import MDS_STORAGE_FIELD, S3_STORAGE_FIELD
from wiki.pages.logic import url_to_wiki_page_in_frontend
from wiki.pages.logic.rank import next_page_rank
from wiki.pages.models.consts import (
    EDITED_PAGE_ACTUALITY_TIMEOUT,
    MARKED_PAGE_ACTUALITY_TIMEOUT,
    ACTUALITY_STATUS,
    COMMENTS_STATUS,
    DB_PAGE_TYPES,
)
from wiki.pages.models.json_props import SerializedPropsMixin
from wiki.sync.connect.base_organization import as_base_organization
from wiki.sync.connect.models import Organization
from wiki.users.models import Group
from wiki.utils.backports.mds_compat import APIError
from wiki.utils.inflections import inflect, inflections
from wiki.utils.s3.storage import S3_STORAGE
from wiki.utils.storage import STORAGE
from wiki.utils.supertag import translit
from wiki.utils.timezone import make_aware_current, now

logger = logging.getLogger(__name__)

YEARS_AGO = make_aware_current(2000, 1, 1)


class RedirectLoopException(Exception):
    pass


class ActivePageManager(models.Manager):
    def translit(self, tag):
        return translit(tag)

    def get_descendants(self, supertag, include_page=False):
        qs = Q(supertag__startswith=supertag + '/', org=get_org())
        if include_page:
            qs |= Q(supertag=supertag, org=get_org())
        return self.filter(qs).order_by('supertag')

    def get_queryset(self):
        return super(ActivePageManager, self).get_queryset().filter(status__gt=0)


def parse_keywords(value):
    value = value.strip()
    if not value:
        return []
    if value.find('\n') != -1:
        value = value.split('\n')
    elif value.find(',') != -1:
        value = value.split(',')
    else:
        value = [value]
    return [x.strip() for x in value]


def upload_to(instance, filename):
    return True


class Page(SerializedPropsMixin, models.Model, IndexableModelMixin):
    TYPES = DB_PAGE_TYPES

    # Translators: translation of each "*Inflections" must contain
    # all inflections separated by | (for Russian) or one word (for English).
    # Order of inflections is subjective, genitive, dative, accusative,
    # ablative, prepositional. Russain example for pages.page:PageInflections:
    # страница|страницы|странице|страницу|страницей|странице
    INFLECTIONS = {
        'P': ugettext_lazy('pages.page:PageInflections'),
        'G': ugettext_lazy('pages.page:ListInflections'),
        'C': ugettext_lazy('pages.page:PageInflections'),
        'W': ugettext_lazy('pages.page:PageInflections'),
    }

    # Storage-stored fields
    _serialized_props = (
        # Page's wiki text
        'body',
        # Wiki text formatted by an old formatter
        'body_r',
        # Table of content created by an old formatter
        'body_toc',
        # Used in "description" HTML meta tag
        'description',
        # Used in "keywords" HTML meta tag, also displayed on the page
        'keywords',
    )

    # Set default value for storage-stored fields
    _serialized_props_defaults = ''

    mds_storage_id = models.FileField(max_length=200, storage=STORAGE, upload_to=upload_to, null=True)
    s3_key = models.FileField(max_length=200, storage=S3_STORAGE, upload_to=upload_to, null=True)

    title = models.CharField(max_length=255, default='')

    tag = models.CharField(max_length=255, db_index=True)
    supertag = models.CharField(max_length=255, db_index=True)

    @property
    def slug(self):  # общепринятый термин для "supertag"
        return self.supertag

    owner = models.ForeignKey(settings.AUTH_USER_MODEL, related_name='pages', null=True, on_delete=models.CASCADE)
    last_author = models.ForeignKey(
        settings.AUTH_USER_MODEL, related_name='changed_pages', null=True, on_delete=models.CASCADE
    )
    page_type = models.CharField(max_length=1, choices=TYPES.choices(), default=TYPES.PAGE)

    redirects_to = models.ForeignKey(
        'self', related_name='redirected_from', blank=True, null=True, default=None, on_delete=models.CASCADE
    )
    lang = models.CharField(max_length=2, default='')
    created_at = models.DateTimeField(default=now)

    authors = models.ManyToManyField(
        settings.AUTH_USER_MODEL,
        blank=True,
    )

    authors_groups = models.ManyToManyField(
        Group,
        blank=True,
    )

    is_official = models.BooleanField(default=False)

    # Do not add auto_now=True to modified_at!
    # Because Page's objects are often saved without change of page
    # as wiki entity, e.g. when 'files' counter has been updated
    modified_at = models.DateTimeField()

    # По этой дате индексатор получает список измененных страниц. Может обновляться не только при редактировании
    # страницы, но при изменении метаданных, например, отнаследованных от родительской страницы прав доступа. Поэтому
    # вынесено в отдельный атрибут.
    # По умолчанию - дата в прошлом, в методе save() подменяется на modified_at, если меньше этого значения.
    modified_at_for_index = models.DateTimeField(default=YEARS_AGO, db_index=True)

    # Counter of page's comments
    comments = models.PositiveIntegerField(default=0)

    # Counter of page's attached files
    files = models.PositiveIntegerField(default=0)

    # Does page block all nested urls?
    # E.g. page /killa/gorilla with is_blocking=True will be resolved for
    # /killa/gorilla/* URLs.
    is_blocking = models.BooleanField(default=False)

    # 0 - неактивна, 1 - активна. Неактивность это полный аналог "удалена".
    status = models.PositiveIntegerField(default=1)

    # Флаг opened_to_external_flag показывает, что страница доступна внешним
    # консультантам. Если None, то статус неизвестен. Пересчитывается таском
    # celery wiki.pages.tasks.access_status
    opened_to_external_flag = models.NullBooleanField(default=None)

    # Версия форматтера, которым сохранена страница.
    formatter_version = models.CharField(max_length=50, default='', verbose_name='formatter')

    # статус актуальности страницы, может быть равен одной из констант
    # ACTUALITY_STATUS.actual, ACTUALITY_STATUS.obsolete, ACTUALITY_STATUS.unspecified,
    # но НЕ ACTUALITY_STATUS.possibly_obsolete (этот статус вычисляется по условию на лету)
    actuality_status = models.PositiveIntegerField(choices=ACTUALITY_STATUS, default=ACTUALITY_STATUS.unspecified)

    actuality_marked_at = models.DateTimeField(null=True, verbose_name='момент смены статуса актуальности')

    is_documentation = models.BooleanField(default=False)

    org = models.ForeignKey(Organization, blank=True, null=True, default=None, on_delete=models.CASCADE)

    # ссылка на исходную страницу, с которой был сделан клон
    reference_page = models.ForeignKey(
        'self', related_name='cloned_pages', blank=True, null=True, default=None, on_delete=models.SET_NULL
    )

    is_autogenerated = models.BooleanField(default=False)

    # Статус доступности комментариев на странице или во всем кластере
    comments_status = models.PositiveIntegerField(choices=COMMENTS_STATUS, default=COMMENTS_STATUS.enabled)

    with_new_wf = models.BooleanField(default=True)

    # флаг, определяющий режим страницы "Только для чтения". Если флаг установлен, то страница будет недоступна для
    # редактирования.
    is_readonly = models.BooleanField(default=False)

    # дата и время переключения признака используемого на странице форматера
    wf_switched_at = models.DateTimeField(null=True)

    # автор переключения признака with_new_wf
    wf_switched_author = models.ForeignKey(
        settings.AUTH_USER_MODEL, related_name='switched_author', null=True, on_delete=models.CASCADE
    )

    # This property contains a previous value of "body" property
    # when it's has been changed. Used in wiki.pages.api.save_page.
    _old_body = None

    objects = models.Manager()
    active = ActivePageManager()

    rank = models.CharField(max_length=255)

    depth = models.PositiveIntegerField(db_index=True)

    background_id = models.PositiveSmallIntegerField(blank=True, null=True)

    body_size = models.PositiveIntegerField(null=True)

    @property
    def _storage_field_name(self):
        if self.s3_key:
            # Страница уже сохранена в S3
            return S3_STORAGE_FIELD

        if self.mds_storage_id:
            # Страница уже сохранена в MDS
            return MDS_STORAGE_FIELD

        if waffle.switch_is_active(S3_DEFAULT_STORAGE):
            return S3_STORAGE_FIELD

        return MDS_STORAGE_FIELD

    @staticmethod
    def get_url(supertag):
        """
        Вернуть path без query до вики-страницы.

        Предназначена для использования в джанго-верстке и не в АПИ.
        """
        return six.text_type('/' + supertag)

    @property
    def url(self):
        """
        Урл относительно корня сервиса до страницы на фронтэнде.

        Предназначено для использования в джанго-верстке и не в АПИ.
        """
        return Page.get_url(self.supertag)

    @property
    def excluded_from_search_index(self):
        return hasattr(self, 'searchexclude')  # забавно, но это Django-recommended way

    @property
    def absolute_url(self):
        """
        Абсолютный URL до фронтэнда.

        @rtype: basestring
        """
        return url_to_wiki_page_in_frontend(self)

    @property
    def absolute_url_before_deletion(self):
        """
        Абсолютный URL до фронтэнда.

        @rtype: basestring
        """
        return url_to_wiki_page_in_frontend(self, restore_if_deleted=True)

    @property
    def comments_count(self):
        return self.comments

    @property
    def is_comments_enabled(self):
        return self.comments_status == COMMENTS_STATUS.enabled

    def get_keywords(self):
        """
        Вернуть список ключевых слов на странице.

        @rtype: list
        """
        try:
            page_keywords = self.keywords
        except APIError:
            # storage errors
            logger.exception('There is no Page in storage:page id "%s"', str(self.mds_storage_id))
            return []
        if not page_keywords:
            return []
        return parse_keywords(page_keywords)

    def set_keywords(self, value):
        self.keywords = '\n'.join(value)

    keywords_list = property(get_keywords, set_keywords, doc='access to old PHP keywords')

    @property
    def descendants(self):
        with org_ctx(self.org):
            return Page.active.filter(supertag__startswith=self.supertag + '/', org=get_org()).order_by('supertag')

    @property
    def parent(self):
        from wiki.pages.logic import hierarchy

        return hierarchy.get_parent(self)

    @property
    def nearest_parent(self):
        from wiki.pages.logic import hierarchy

        return hierarchy.get_nearest_existing_parent(self)

    @property
    def supertag_before_deletion(self):
        if self.status != 0:
            return self.supertag
        # при удалени добавляется рандомная секция слева
        return '/'.join(self.supertag.split('/')[1:])

    @property
    def ancestors(self):
        supertags_chain = self.get_supertags_chain()
        with org_ctx(self.org):
            return Page.active.filter(supertag__in=supertags_chain, org=get_org()).order_by('supertag')

    def get_authors(self):
        return self.authors.all()

    def is_all_authors_dismissed(self):
        return not self.authors.filter(staff__is_dismissed=False).exists()

    def redirect_target(self):
        """Вернуть последний редирект в цепочке редиректов от этой страницы

        @rtype: Page
        @raise: RedirectLoopException

        """
        for page in self.redirect_chain():
            pass
        return page

    def redirect_chain(self):
        """
        Генератор цепочки страниц, которые являются редиректами начиная с текущей.

        Цепочка никогда не зацикливается. Всегда возвращает хотя бы саму себя.
        Не возвращает удаленные страницы.

        @raise: RedirectLoopException
        """
        yield self

        pages = set([self.pk])
        target_page = self
        while target_page.has_redirect():
            if len(pages) == 500:
                # 500 редиректов похоже на намеренный поиск уязвимости
                return
            target_page = target_page.redirects_to
            if not target_page.status:
                return
            yield target_page
            if target_page.pk in pages:
                raise RedirectLoopException('Redirect loop, pages ids: %s' % str(pages))
            pages.add(target_page.pk)

    def has_redirect(self):
        return self.redirects_to is not None

    def inflect(self, inflection):
        """
        Returns inflected name of current page's type in current locale.
        Possible value of inflection param is 'subjective', 'genitive',
        'dative', 'accusative', 'ablative' or 'prepositional'.
        """
        return inflect(self.INFLECTIONS[self.page_type], inflection)

    def __setattr__(self, name, value):
        """
        Evil hack for getting old page's body in Page.save method.
        Allow to get old body for creating change's diff for
        generation of "edition" log event.
        """
        if name == 'body':
            self.body_size = len(value)
            if self._old_body is None:
                self._old_body = self.body
        return super(Page, self).__setattr__(name, value)

    def __getattr__(self, name):
        """
        Override parent __getattr__ to provide inflections properties
        """
        if name in inflections:
            return self.inflect(name)
        return super(Page, self).__getattr__(name)

    def save(self, **kwargs):
        """
        Override base ORM save method by adding default values for some fields
        and ability to force insertion or update detection.
        """
        if not self.id:
            kwargs['force_insert'] = True
            if not self.created_at:
                self.created_at = now()
        else:
            kwargs['force_update'] = True

        # Обновляем modified_at_for_index, чтобы страница переиндексировалась при обновлении.
        if self.modified_at_for_index < self.modified_at:
            self.modified_at_for_index = self.modified_at

        if self.is_props_changed():
            self.reference_page = None

        self.depth = len(self.supertag.split('/'))

        # Для новых страниц расчитываем значение rank,
        # используемое при сортировке в дереве навигации
        if not self.id:
            self.rank = next_page_rank(self.supertag, self.depth, as_base_organization(self.org))

        result = super(Page, self).save(**kwargs)

        return result

    @property
    def breadcrumbs(self) -> Union[List[BreadcrumbSchema], List[Dict[str, Any]]]:
        """
        Возвращает информацию о родителях в порядке от наиболее близкого к корню в виде массива вида:
        [
            {
                'name': '...',  # Кусок тега
                'tag': '...',  # Тег
                'url': '...',  # относительный УРЛ страницы
                'title': '...'  # Заголовок
                'is_active': false # Флаг наличия страницы
            },
        ]
        """

        from wiki.pages.logic.breadcrumbs import get_breadcrumbs_compat

        return get_breadcrumbs_compat(self)

    def get_supertags_chain(self, include_self=False):
        """
        Вернуть супертеги всех родителей, в т.ч. виртуальных, которых в базе нет.

        Родители упорядочены от корня к supertag.

        @param include_last: включать сам supertag или нет.
        """
        from wiki.pages.logic import hierarchy

        return hierarchy.get_supertags_chain(self, include_self=include_self)

    @property
    def is_active(self):
        """
        Return True if page is active
        and False if page was already marked as deleted.

        @rtype: bool
        """
        return not self.status == 0

    def get_page_version(self):
        from wiki.pages.models.revision import Revision

        version = Revision.objects.filter(page_id=self.id).aggregate(max_id=Max('pk'))['max_id']
        if version is None:
            # ситуация не штатная, но по факту такое в вики бывает на тестинге
            raise ValueError('Could not find a single revision, unhandled error')
        return str(version)

    @property
    def has_manual_actuality_mark(self):
        """
        True если актуальность/устаревшесть были указаны пользователем.
        """
        return self.actuality_status in (ACTUALITY_STATUS.actual, ACTUALITY_STATUS.obsolete)

    @property
    def realtime_actuality_status(self):
        """
        Преобразует хранящийся в базе статус актуальности с учётом условия "потенциально устаревшей страницы"
        """
        if self.actuality_status == ACTUALITY_STATUS.obsolete:
            return ACTUALITY_STATUS.obsolete

        NOW = now()
        modified_long_ago = NOW - self.modified_at > EDITED_PAGE_ACTUALITY_TIMEOUT
        marked_actual_long_ago = NOW - (self.actuality_marked_at or NOW) > MARKED_PAGE_ACTUALITY_TIMEOUT

        if self.actuality_status == ACTUALITY_STATUS.unspecified:
            if modified_long_ago:
                return ACTUALITY_STATUS.possibly_obsolete
            else:
                return ACTUALITY_STATUS.actual
        elif self.actuality_status == ACTUALITY_STATUS.actual and marked_actual_long_ago and modified_long_ago:
            return ACTUALITY_STATUS.possibly_obsolete

        return ACTUALITY_STATUS.actual

    def __str__(self):
        return 'Page %s: "%s"' % (self.id, self.supertag)

    class Meta:
        db_table = 'pages_page'
        app_label = 'pages'

        verbose_name = 'страница'
        verbose_name_plural = 'страницы'

        unique_together = (('supertag', 'org'),)

    def get_metadata(self):
        return {
            'url': self.url,
            'created_at': int(self.created_at.timestamp()),
            'modified_at': int(self.modified_at.timestamp()),
            'authors': [{'uid': a.get_uid(), 'cloud_uid': a.get_cloud_uid()} for a in self.get_authors()],
            'is_obsolete': (self.actuality_status == ACTUALITY_STATUS.obsolete),
            'slug': self.supertag,
            'type': 'page',
            'page_type': self.page_type,
        }

    def get_document(self):
        return {'title': self.title, 'body': self.body, 'keywords': self.get_keywords()}

    def get_search_uuid(self):
        if self.page_type in {self.TYPES.PAGE, self.TYPES.WYSIWYG}:
            return f'wp:{self.id}'
        if self.page_type == self.TYPES.GRID:
            return f'wg:{self.id}'
        raise ValueError(f'can not get search uuid for page {self.id} with type {self.page_type}')

    def get_acl_subject(self):
        return self

    def get_model_org(self):
        return self.org

    def get_children_files(self):
        return self.file_set.all()

    def get_children_pages(self):
        return Page.objects.filter(supertag__startswith=self.supertag + '/', org=self.get_model_org())


class PageRename(models.Model):
    """
    Временная таблица для переименования плохих тегов на всех страницах.
    """

    page = models.OneToOneField(to=Page, on_delete=models.CASCADE)
    old_tag = models.CharField(max_length=255, db_index=True)
    old_supertag = models.CharField(max_length=255, db_index=True)
    new_tag = models.CharField(max_length=255, db_index=True)
    new_supertag = models.CharField(max_length=255, db_index=True)

    class Meta:
        app_label = 'pages'
