
import logging

from django.conf import settings
from django.db import models

from wiki.pages.constants import MDS_STORAGE_FIELD, S3_STORAGE_FIELD
from wiki.pages.models.json_props import SerializedPropsMixin
from wiki.utils.s3.storage import S3_STORAGE
from wiki.utils.storage import STORAGE

logger = logging.getLogger(__name__)


class RevisionManager(models.Manager):
    # TODO: перенести в модель Revision как @classmethod

    def produce_from_page(self, page):
        new_rev = self.model(
            page=page,
            title=page.title[:250],
            created_at=page.modified_at,
            author=page.last_author,
            formatter_version=page.formatter_version,
            body_size=page.body_size,
        )

        if page.mds_storage_id:
            new_rev.mds_storage_id = page.mds_storage_id
        elif page.s3_key:
            new_rev.s3_key = page.s3_key
        else:
            logger.error(f'You are creating revision for page "{page}" without storage')

        return new_rev

    def create_from_page(self, page):
        new_rev = self.produce_from_page(page)
        new_rev.save()
        return new_rev


def upload_to(instance, filename):
    return True


class Revision(SerializedPropsMixin, models.Model):
    _serialized_props = ('body', 'description', 'keywords')
    page = models.ForeignKey('Page', on_delete=models.CASCADE)
    author = models.ForeignKey(settings.AUTH_USER_MODEL, related_name='revision_author_set', on_delete=models.CASCADE)
    title = models.CharField(max_length=250, default='')
    created_at = models.DateTimeField(db_index=True)
    mds_storage_id = models.FileField(max_length=200, storage=STORAGE, upload_to=upload_to, null=True)
    s3_key = models.FileField(max_length=200, storage=S3_STORAGE, upload_to=upload_to, null=True)

    # Версия форматтера, которым сохранена страница.
    formatter_version = models.CharField(max_length=50, default='1.0')

    body_size = models.PositiveIntegerField(null=True)

    @property
    def _storage_field_name(self):
        if self.s3_key:
            return S3_STORAGE_FIELD

        return MDS_STORAGE_FIELD

    @property
    def url(self):
        # FIXME: здесь хак. сделан для решения тикета WIKI-4776 в стартреке
        return self.page.url

    def get_page_version(self):
        return str(self.id)

    def __str__(self):
        return '{class_name} {id} of page "{supertag}" made by {username} at {date}'.format(
            class_name=self.__class__.__name__,
            id=self.id,
            supertag=self.page.supertag,
            username=self.author.username if self.author else 'unknown',
            date=self.created_at,
        )

    objects = RevisionManager()

    class Meta:
        app_label = 'pages'
