from typing import List
from datetime import timedelta
from django.db import connection
from django.conf import settings

from wiki.acl.check_access import assert_has_access_by_slug, has_access
from wiki.acl.consts import Action
from wiki.api_v2.public.pages.schemas import NavigationTreeSchema, NavigationTreeNode
from wiki.notifications.generators.base import EventTypes
from wiki.notifications.models import PageEvent
from wiki.pages.access import get_bulk_access_status, ACCESS_COMMON, ACCESS_RESTRICTED, ACCESS_UNLIMITED
from wiki.pages.constants import PageOrderPosition
from wiki.pages.exceptions import UpdatePagePositionError
from wiki.pages.logic.rank import calculate_page_rank
from wiki.pages.models import Page
from wiki.pages.models.consts import DB_PAGE_TYPES_MAP
from wiki.sync.connect.org_ctx import org_ctx
from wiki.utils import timezone
from wiki.utils.db import namedtuple_fetchall

EXCLUDED_SLUGS = settings.NAVIGATION_TREE_EXCLUDED_SLUGS


def update_node_position(slug: str, position: PageOrderPosition, next_to_slug: str, organization, user):
    if next_to_slug == slug:
        raise UpdatePagePositionError('The moved page and next_to_slug are the same')

    # next_to_slug и переносимая страница в одном и том же кластере
    page_cluster = '/'.join(slug.split('/')[:-1])
    expected_cluster = '/'.join(next_to_slug.split('/')[:-1])
    if not page_cluster == expected_cluster:
        raise UpdatePagePositionError('The moved page parent cluster is different from next_to_slug cluster')

    # Есть доступ для кластера, в котором происходит перемещение
    assert_has_access_by_slug(user, organization, page_cluster, Action.READ)

    rank = calculate_page_rank(position, next_to_slug, organization)
    page = organization.get_active_pages().get(supertag=slug)
    old_rank = page.rank
    page.rank = rank
    page.save()

    PageEvent(
        page=page,
        event_type=EventTypes.change_position,
        author=user,
        timeout=timezone.now() + timedelta(minutes=5),
        notify=False,
        meta={
            'position': position,
            'next_to_slug': next_to_slug,
            'new_rank': rank,
            'old_rank': old_rank,
        },
    ).save()


class NavigationTree:
    def __init__(self, slug: str, user, organization, page_id: int = 0, page_size: int = 50):
        self.slug = slug
        self.user = user
        self.organization = organization
        self.limit = page_size
        self.offset = (page_id - 1) * page_size

    def build_tree(self) -> NavigationTreeSchema:
        try:
            page = self.organization.get_active_pages().get(supertag=self.slug)

            page_title = page.title
            page_type = page.page_type
            if not has_access(self.user, self.organization, page, Action.READ):
                page_title = page.supertag
                page_type = None

            node = NavigationTreeNode(
                page_id=page.id,
                slug=page.supertag,
                title=page_title,
                page_type=DB_PAGE_TYPES_MAP.get(page_type),
                children_count=0,
                is_missing=False,
            )
        except Page.DoesNotExist:
            node = NavigationTreeNode(
                page_id=None,
                slug=self.slug,
                title=self.slug,
                page_type=None,
                children_count=0,
                is_missing=True,
            )

        result = NavigationTreeSchema(node=node, children=[], has_next=False)

        if self.slug in EXCLUDED_SLUGS:
            return result

        slug_parts = self.slug.split('/')

        org = self.organization.as_django_model()
        org_filter = f'= {org.id}' if org is not None else 'IS NULL'

        sql_query = (
            "WITH slugs as (SELECT %s || SPLIT_PART(supertag, '/', %s) as path, COUNT(1) as count "
            'FROM pages_page WHERE supertag LIKE %s AND status > 0 AND org_id {org_filter} GROUP BY path) '
            'SELECT p.id, p.supertag as page_supertag, p.page_type, p.title, '
            'slugs.path as supertag, slugs.count from slugs '
            'LEFT JOIN pages_page p ON p.supertag = slugs.path '
            'WHERE (p.status > 0 AND p.org_id {org_filter}) or p.id is NULL '
            'ORDER BY p.rank, p.created_at LIMIT %s OFFSET %s'
        ).format(org_filter=org_filter)

        # Выбираем количество записей на 1 больше, чем лимит
        # чтобы проверить, есть ли данные сверх лимита и заполнить
        # соответствующее поле has_next в ответе API
        sql_params = [f'{self.slug}/', len(slug_parts) + 1, f'{self.slug}/%', self.limit + 1, self.offset]

        with connection.cursor() as cursor:
            cursor.execute(sql_query, sql_params)
            rows = namedtuple_fetchall(cursor)

            if len(rows) > self.limit:
                result.has_next = True
                rows = rows[: self.limit]

            slugs_with_access = {row.supertag for row in self._filter_by_access(rows)}
            for row in rows:
                page_title = row.title
                page_type = row.page_type
                if row.supertag not in slugs_with_access:
                    page_title = row.supertag
                    page_type = None

                result.children.append(
                    NavigationTreeNode(
                        page_id=row.id,
                        slug=row.supertag,
                        title=page_title,
                        page_type=DB_PAGE_TYPES_MAP.get(page_type),
                        children_count=row.count - 1 if row.id else row.count,
                        is_missing=False if row.id else True,
                    )
                )

        result.node.children_count = len(result.children)
        return result

    def _filter_by_access(self, page_rows: List[str]) -> List[str]:
        with org_ctx(self.organization.as_django_model()):
            access_statuses = get_bulk_access_status(
                tags=[page.supertag for page in page_rows],
                user=self.user,
            )
        valid_access = {ACCESS_COMMON, ACCESS_RESTRICTED, ACCESS_UNLIMITED}
        return [page for page in page_rows if access_statuses[page.supertag] in valid_access]
