from time import mktime

from django.conf import settings

from wiki.actions.classes.base_widget_action import WikiWidgetAction
from wiki.api_core.serializers.users import UserSerializer
from wiki.files.models import File
from wiki.pages.access import ACCESS_COMMON
from wiki.pages.logic.subpages import get_subpages
from wiki.pages.models import Page
from wiki.utils.timezone import make_aware_current, make_naive_current

# TODO: После удаления старого экшена {{tree}} и старого хендлера Tree удалить соседний модуль pages_tree.


class SortBy:
    TITLE = 'title'
    CLUSTER = 'cluster'
    CREATED_AT = 'created_at'
    MODIFIED_AT = 'modified_at'

    CHOICES = (TITLE, CLUSTER, CREATED_AT, MODIFIED_AT)
    DEFAULT = TITLE


class Sort:
    ASC = 'asc'
    DESC = 'desc'

    CHOICES = (ASC, DESC)
    DEFAULT = ASC


DISTANT_FUTURE = make_aware_current(9999, 1, 1)


class PagesTree(object):
    def __init__(
        self,
        root_supertag,
        user,
        expand_subtree_url_builder,
        depth=None,
        show_redirects=True,
        show_grids=True,
        show_files=False,
        show_owners=False,
        show_titles=True,
        show_created_at=False,
        show_modified_at=False,
        sort_by=SortBy.DEFAULT,
        sort=Sort.DEFAULT,
        from_yandex_server=False,
        authors=None,
    ):
        """
        Построить дерево страниц для указанного супертэга.

        https://wiki.yandex-team.ru/wiki/components/pages-tree/

        @param root_supertag: Супертэг корневой страницы дерева.
        @type root_supertag: str

        @type user: User

        @param expand_subtree_url_builder: Конструктор URL разворачивания поддеревьев.
        @type expand_subtree_url_builder: ExpandSubtreeUrlBuilder

        @param depth: Максимальная глубина страниц в дереве.
        @type depth: int

        @param show_redirects: Если True, то редиректы будут включены в дерево.
        @type show_redirects: bool

        @param show_grids: Если True, то гриды будут включены в дерево.
        @type show_grids: bool

        @param show_files: Если True, то в дерево будут добавлены файлы, прикрепленные к страницам.
        @type show_files: bool

        @param show_owners: Если True, то в дерево будут добавлены авторы страниц.
        @type show_owners: bool

        @param show_titles: Если True, то в дерево будут добавлены заголовки страниц.
        @type show_titles: bool

        @param show_created_at: Если True, то в дерево будут добавлены даты созданий страниц.
        @type show_created_at: bool

        @param show_modified_at: Если True, то в дерево будут добавлены даты модификаций страниц.
        @type show_modified_at: bool

        @param sort_by: 'title' | 'cluster' | 'created_at' | 'modified_at'
        Атрибут страницы, по которому производится сортировка.
        @type sort_by: str

        @param sort: 'asc' | 'desc'
        Сортировка по возрастанию/убыванию
        @type sort: str

        @type from_yandex_server: bool

        @param authors: показывать только страницы, где автором является один из пользователей из переданного списка
        @type authors: list
        """
        self.root_supertag = root_supertag or settings.MAIN_PAGE
        self.user = user
        self.expand_subtree_url_builder = expand_subtree_url_builder
        self.depth = depth
        self.show_redirects = show_redirects
        self.show_grids = show_grids
        self.show_files = show_files
        self.show_owners = show_owners
        self.show_titles = show_titles
        self.show_created_at = show_created_at
        self.show_modified_at = show_modified_at
        self.from_yandex_server = from_yandex_server
        self.authors = authors

        self.subpages_sort_key = self._determine_sort_key(sort_by)
        self.sort_reverse = sort == Sort.DESC

        self.data = self._build_tree()

    def _build_tree(self):
        # Мы извлекаем из базы данных страницы с глубиной на единицу больше.
        # Это нужно, чтобы выяснить, есть ли подстраницы у страниц с максимальной глубиной.
        # Если подстраницы есть и хотя бы одна из них видна пользователю,
        # то нужно показать кнопку разворачивания поддерева.
        max_depth = self.depth and self.depth + 1

        self.subpages, self.page_pk_to_access, self.limit_exceeded = get_subpages(
            page_supertag=self.root_supertag,
            user=self.user,
            max_depth=max_depth,
            limit=5_000,
            show_owners=self.show_owners,
            from_yandex_server=self.from_yandex_server,
            authors=self.authors,
        )

        # Строим дерево глубиной depth + 1, в котором присутствуют все страницы.
        #
        # Каждый узел получившегося дерева состоит из двух атрибутов:
        #
        # 'page' - модель страницы из базы данных, или None, если она не существует,
        # 'subpages' - словарь {cluster -> subpages},
        # где кластер - суффикс супертэга страницы (то, что находится за самым правым '/')
        #
        # Пример. Пусть у нас есть страницы со следующими супертэгами:
        #
        # wiki/api
        # wiki/api/old
        # wiki/api/old/xml/get_access
        # wiki/api/new
        # wiki/api/new/html
        # wiki/api/new/json
        #
        # Построим дерево с корнем wiki/api:
        #
        #   {
        #       'page': <page: wiki/api>,
        #       'subpages': {
        #           'old': {
        #               'page': <page: wiki/api/old>,
        #               'subpages': {
        #                   'xml': {
        #                       'page': None, // страницы с супертэгом wiki/api/old/xml не существует
        #                       'subpages': {
        #                           'get_access': {
        #                               'page': <page: wiki/api/old/xml/get_access>,
        #                               'subpages': {},
        #                           },
        #                       },
        #                   },
        #               },
        #           },
        #           'new': {
        #               'page': <page: wiki/api/new>,
        #               'subpages': {
        #                   'html': {
        #                       'page': <page: wiki/api/new/html>,
        #                       'subpages': {},
        #                   },
        #                   'json': {
        #                       'page': <page: wiki/api/new/json>,
        #                       'subpages': {},
        #                   },
        #               },
        #           },
        #       },
        #   }
        #
        self.tree = self._build_raw_tree(self.subpages, self.root_supertag)

        # Определим невидимые пользователю типы страниц.
        #
        # Страница видима, если она не закрыта от пользователя согласно правилам доступа,
        # существует и удовлетворяет критериям show_redirects и show_grids.
        self.invisible_page_types = self._determine_invisible_page_types(self.show_redirects, self.show_grids)

        # Невидимые страницы впоследствии будут удалены из дерева.
        # Но есть исключение, когда невидимые пользователю страницы все же присутствуют в дереве:
        # если в дереве с ограничением глубины depth есть видимая страница на глубине меньше или равной depth + 1,
        # то вся цепочка ее родительских страниц вплоть до корневой страницы,
        # включая невидимые страницы, попадает в дерево.

        # Обходим узлы дерева. В процессе обхода происходит следующее:
        #
        # 1) Узлы дерева размечаются атрибутами:
        # 'type' - тип страницы,
        # 'subpages_count' - число существующих подстраниц на следующем уровне глубины.
        #
        # 2) Если узел дерева находится на максимальной глубине, удаляются все его подстраницы и проставляется
        # атрибут 'is_folded', который равен True, если у узла были видимые подстраницы.
        #
        # 3) Если узел дерева не находится на максимальной глубине, то проставляется атрибут 'is_folded'=False,
        # и у узла удаляются все поддеревья без видимых страниц.
        self._traverse_tree()

        # Загрузим прикрепленные к страницам файлы.
        self.page_pk_to_files = self._get_files()

        # Построим итоговое дерево с данными, нужными в UI.
        #
        # Структура узла итогового дерева:
        #
        #   {
        #       'cluster': <page_cluster>,
        #       'url': <page_url>,
        #       'type': <page_type>,
        #       'title': <page_title> (optional),
        #       'created_at': <page_created_at> (optional),
        #       'modified_at': <page_modified_at> (optional),
        #       'authors': [<page_author>,] (optional),
        #       'files': [...] (optional),
        #       'expand_url': <expand_subtree_handler_url> (optional),
        #       'subpages_count': <int>,
        #       'subpages': [...],
        #   }
        #
        # 'url' - строится по цепочке кластеров.
        # 'title' - присутствует, если show_titles=True и страница не закрытая.
        # 'created_at' - присутствует, если show_created_at=True и страница не закрытая.
        # 'modified_at' - присутствует, если show_modified_at=True и страница не закрытая.
        # 'authors' - присутствует, если show_owners=True и страница не закрытая.
        # 'files' - присутствует, если show_files=True, страница не закрытая и у нее есть прикрепленные файлы.
        # 'expand_url' - присутствует, если в соответствующем узле 'is_folded'=True
        # 'subpages' - список подстраниц
        #
        data = self._build_final_node(self.tree, clusters=tuple())

        # В корневой узел добавляются два дополнительных атрибута:
        #
        # 'limit_exceeded' - если True, то число подстраниц оказалось больше,
        # чем лимит извлечения из базы данных, и нужно уведомить об этом пользователя.
        #
        # 'expand_all_url' - URL для разворачивания всего дерева.

        data['limit_exceeded'] = self.limit_exceeded

        # Если дерево отображается с ограничением по глубине, мы не можем точно знать,
        # развернуто оно до конца или нет, поскольку у нас есть только подстраницы
        # с максимальной глубиной depth + 1. Мы можем быть уверены, что все дерево развернуто полностью
        # только в случае, если ограничения по глубине нет. Во всех остальных случаях мы будем
        # отображать кнопку "Показать все подстраницы", хотя в некоторых случаях
        # после ее нажатия новых подстраниц не появится - дерево уже было целиком развернуто.
        if self.depth is not None:
            data['expand_all_url'] = self.expand_subtree_url_builder.build_url(self.root_supertag)

        return data

    @staticmethod
    def _build_raw_tree(pages, root_supertag):
        tree = {'subpages': {}, 'page': None}
        root_prefix_len = len(root_supertag) + 1

        for page in pages:
            supertag_suffix = page.supertag[root_prefix_len:]
            if supertag_suffix:
                clusters = supertag_suffix.split('/')
            else:
                clusters = []

            node = tree
            for cluster in clusters:
                if cluster not in node['subpages']:
                    node['subpages'][cluster] = {'subpages': {}, 'page': None}
                node = node['subpages'][cluster]
            node['page'] = page

        return tree

    @staticmethod
    def _determine_invisible_page_types(show_redirects, show_grids):
        types = ['C', 'N']
        if not show_redirects:
            types.append('R')
        if not show_grids:
            types.append('G')
        return types

    @staticmethod
    def _determine_page_type(page, page_pk_to_access):
        if not page:
            return 'N'
        access = page_pk_to_access.get(page.pk, ACCESS_COMMON)
        if access < 1:
            return 'C'
        if page.has_redirect():
            # Если страница - редирект на закрытую страницу,
            # то мы все равно считаем ее редиректом.
            return 'R'
        elif page.page_type == Page.TYPES.GRID:
            return 'G'
        elif page.page_type == Page.TYPES.PAGE:
            return 'P'
        elif page.page_type == Page.TYPES.CLOUD:
            return 'X'
        elif page.page_type == Page.TYPES.WYSIWYG:
            return 'W'
        else:
            raise ValueError('Page "%s" has unknown type "%s"' % (page.supertag, page.page_type))

    @staticmethod
    def _determine_sort_key(sort_by):
        def _get_date_sort_attr(node):
            if node['page']:
                return getattr(node['page'], sort_by)
            else:
                return DISTANT_FUTURE

        def _get_title_sort_attr(node, cluster):
            if node['page'] and node['type'] != 'C':
                return node['page'].title
            else:
                return cluster

        if sort_by in (SortBy.CREATED_AT, SortBy.MODIFIED_AT):
            # Сортируем сначала по дате, несуществующие страницы
            # попадают в конец и сортируются по кластеру.
            return lambda cluster, node: (_get_date_sort_attr(node), cluster)
        elif sort_by == SortBy.TITLE:
            # Сортируем обычные страницы по заголовку,
            # а несуществующие и закрытые - по кластеру.
            return lambda cluster, node: _get_title_sort_attr(node, cluster)
        else:
            # Сортируем все страницы по кластеру.
            return lambda cluster, node: cluster

    def _traverse_tree(self):
        if not self._traverse_node(self.tree, 0):
            self.tree['subpages'] = {}

    def _traverse_node(self, node, current_depth):
        node['type'] = self._determine_page_type(node['page'], self.page_pk_to_access)

        has_visible_childs = False
        subpages_count = 0
        child_clusters_to_remove = []

        for cluster, subnode in node['subpages'].items():
            if self._traverse_node(subnode, current_depth + 1):
                has_visible_childs = True
            else:
                child_clusters_to_remove.append(cluster)
            if subnode['type'] != 'N':
                subpages_count += 1

        node['subpages_count'] = subpages_count

        visible = node['type'] not in self.invisible_page_types
        include = visible or has_visible_childs

        if self.depth and current_depth > self.depth:
            return include

        if self.depth and current_depth == self.depth:
            # Удаляем страницы с глубиной depth + 1.
            node['is_folded'] = has_visible_childs
            node['subpages'] = {}
        else:
            node['is_folded'] = False

            # Небольшая оптимизация, позволяющая не делать лишних удалений из словаря:
            # если страница сама не будет включена в дерево, то нет смысла удалять ее поддеревья -
            # она вместе с ними всеми будет удалена родительской страницей.
            if include:
                for cluster in child_clusters_to_remove:
                    del node['subpages'][cluster]

        return include

    def _get_files(self):
        if not self.show_files:
            return {}

        # Файлы показываем только у незакрытых страниц.
        non_closed_pages = [page for page in self.subpages if self.page_pk_to_access[page.pk] > 0]

        files_list = (
            File.active.filter(page__in=non_closed_pages)
            .values(
                'page__pk',
                'page__supertag',
                'name',
                'url',
            )
            .order_by('name')
        )

        file_url_pattern = '{page_url}/.files/{file_url}?download=1'

        page_pk_to_files = {}
        for file_info in files_list:
            if file_info['name'] is not None:
                page_pk = file_info['page__pk']
                page_url = '/' + file_info['page__supertag']

                file_name = file_info['name']
                file_url = file_info['url']

                page_pk_to_files.setdefault(page_pk, []).append(
                    {'name': file_name, 'url': file_url_pattern.format(page_url=page_url, file_url=file_url)}
                )

        return page_pk_to_files

    def _build_final_node(self, node, clusters):
        page_data, subpages_count, expand_url = self._get_page_data(node, clusters)

        data_node = {'page': page_data, 'subpages_count': subpages_count, 'subpages': []}

        if expand_url:
            data_node['expand_url'] = expand_url

        # Сортировка подстраниц
        subnodes = sorted(
            iter(node['subpages'].items()),
            key=lambda _cluster__node: self.subpages_sort_key(_cluster__node[0], _cluster__node[1]),
            reverse=self.sort_reverse,
        )

        for cluster, subnode in subnodes:
            data_node['subpages'].append(self._build_final_node(subnode, clusters=clusters + (cluster,)))

        return data_node

    def _get_page_data(self, node, clusters):
        if len(clusters):
            cluster = clusters[-1]
            supertag = self.root_supertag + '/' + '/'.join(clusters)
        else:
            cluster = self.root_supertag.split('/')[-1] if len(self.root_supertag) else ''
            supertag = self.root_supertag

        data = {
            'cluster': cluster,
            'url': '/' + supertag,
            'type': node['type'],
        }

        def date_to_timestamp(date):
            return int(mktime(make_aware_current(make_naive_current(date)).timetuple()))

        page = node['page']
        if page and node['type'] != 'C':
            if self.show_titles:
                data['title'] = page.title
            if self.show_owners:
                data['authors'] = UserSerializer(page.get_authors(), many=True).data
            if self.show_created_at:
                data['created_at'] = date_to_timestamp(page.created_at)
            if self.show_modified_at:
                data['modified_at'] = date_to_timestamp(page.modified_at)

        if page and page.pk in self.page_pk_to_files:
            data['files'] = self.page_pk_to_files[page.pk]

        expand_url = None
        if node['is_folded']:
            expand_url = self.expand_subtree_url_builder.build_url(supertag)

        subpages_count = node['subpages_count']

        return data, subpages_count, expand_url


class ExpandSubtreeUrlBuilder(object):
    """
    Конструктор URL разворачивания поддеревьев.
    """

    encoded_get_params = ''
    protocol_and_hostname = ''

    def __init__(self, get_params, nginx_host):
        """
        @type get_params: dict
        @type nginx_host: str
        """
        # Для разворачивания поддеревьев или всего дерева все параметры остаются как были,
        # кроме depth - глубина выставляется неограниченной.
        self.action_params = {
            'depth': 0,
            'show_redirects': get_params['show_redirects'],
            'show_grids': get_params['show_grids'],
            'show_files': get_params['show_files'],
            'show_owners': get_params['show_owners'],
            'show_titles': get_params['show_titles'],
            'show_created_at': get_params['show_created_at'],
            'show_modified_at': get_params['show_modified_at'],
            'sort_by': get_params['sort_by'],
            'sort': get_params['sort'],
            'authors': get_params.get('authors', ''),
        }
        self.nginx_host = nginx_host

    def build_url(self, supertag):
        """
        Получить абсолютный URL ручки для заданного супертэга.

        @type supertag: str
        """
        self.action_params['page'] = supertag
        return '//{host}{rest_of_url}'.format(
            host=self.nginx_host, rest_of_url=WikiWidgetAction.build_url_with_data('tree', supertag, self.action_params)
        )
