import logging

from django.conf import settings
from wiki.pages.constants import ReservedCreationError, ReservedDeletionError
from wiki.pages.constants import ReservedSupertagAction as Action

logger = logging.getLogger('wiki.api_frontend.tests.pages')


def is_reserved_supertag(supertag, action):
    for params in settings.RESERVED_SUPERTAGS:
        pattern = params['pattern']
        if action in params['actions'] and pattern.search(supertag):
            return True
    return False


no_reserved_protection_attr = '_should_skip_reserved_protection'


def protect_reserved_supertags(sender, **kwargs):
    """
    Receiver of pre_save signal that doesn't let users create or delete
    pages or grids with reserved supertags.

    See wiki.pages.signals and wiki.grids.signals
    """

    page = kwargs['instance']

    # Do nothing if a page has special flag to skip the check
    should_skip = getattr(page, no_reserved_protection_attr, False)
    if should_skip:
        return

    check_creation(page)
    check_deletion(page)


def check_creation(page):
    """Raise error on creation of reserved page"""

    # New page is being created?
    if page.pk is None:

        # Is its address reserved?
        if is_reserved_supertag(page.supertag, action=Action.CREATE):
            error_message = 'supertag {0!r} is reserved, won\'t create'.format(page.supertag)
            raise ReservedCreationError(error_message)


def check_deletion(page):
    """Raise error on deletion of reserved page"""

    if page.pk is None:
        return

    # Get the same page from database to compare changes
    Page = page.__class__
    try:
        original_page = Page.objects.get(pk=page.pk)
    except Page.DoesNotExist:
        return

    # Existing page is being deleted?
    if page.status & 0x1 == 0 and original_page.status & 0x1 == 1:

        # Is its address reserved?
        if is_reserved_supertag(original_page.supertag, action=Action.DELETE):
            error_message = 'supertag {0!r} is reserved, won\'t delete'.format(original_page.supertag)
            raise ReservedDeletionError(error_message)
