from django import dispatch
from django.conf import settings
from django.core.cache import cache
from django.db.models import Q
from django.db.models.signals import post_save, pre_delete, pre_save

from wiki.org import get_org
from wiki.pages.access.cache import invalidate_access_rights
from wiki.pages.models import Access, Page
from wiki.sync.connect.org_ctx import org_ctx
from wiki.utils.supertag import tag_to_supertag

IS_INTRANET = getattr(settings, 'IS_INTRANET', False)

access_changed = dispatch.Signal(providing_args=['page_list'])


def invalidate_cache(sender, page_list, **kwargs):
    condition = None
    org = None
    supertags = []
    for page in page_list:
        if isinstance(page, str):
            supertag = tag_to_supertag(page)
        else:
            supertag = page.supertag
        supertags.append(supertag)

        if isinstance(page, Page):
            org = page.org
        else:
            org = get_org()

        # Нужно также выбрать также все дочерние страницы
        if condition is None:
            condition = Q(supertag__startswith=supertag + '/', org=org)
        else:
            condition = condition | Q(supertag__startswith=supertag + '/', org=org)

    if condition is not None:
        supertags.extend(Page.objects.filter(condition).values_list('supertag', flat=True))

    with org_ctx(org, raise_on_empty=False):
        invalidate_access_rights(supertags)


access_changed.connect(invalidate_cache)


def on_access_saved(sender, instance, **kwargs):
    invalidate_cache(sender, [instance.page])


post_save.connect(on_access_saved, sender=Access)

if IS_INTRANET:
    from wiki.intranet.models import GroupMembership
    from wiki.pages.access.cache import all_groups_cached_key

    def reset_group_cache(sender, instance, **kwargs):
        """Сбрасывает запись в memcached о том, в каких группах состоит пользователь"""
        if not instance.staff:
            return
        cache.delete_many([all_groups_cached_key(instance.staff)])

    pre_save.connect(reset_group_cache, sender=GroupMembership)
    pre_delete.connect(reset_group_cache, sender=GroupMembership)
